<?php

namespace App\Console\Commands\ShipStation;

use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\SalesOrderFulfillment;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Arr;

class GetShipments extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:shipstation:get-shipments
                            {--recipientName= : Returns shipments shipped to the specified recipient name.}
                            {--recipientCountryCode= : Returns shipments shipped to the specified country code.}
                            {--orderNumber= : Returns shipments whose orders have the specified order number.}
                            {--orderId= : Returns shipments whose orders have the specified order ID.}
                            {--carrierCode= : Returns shipments shipped with the specified carrier.}
                            {--serviceCode= : Returns shipments shipped with the specified shipping service.}
                            {--serviceCode= : Returns shipments shipped with the specified shipping service.}
                            {--trackingNumber= : Returns shipments with the specified tracking number.}
                            {--createDateStart= : Returns shipments created on or after the specified createDate.}
                            {--createDateEnd= : Returns shipments created on or before the specified createDate.}
                            {--shipDateStart= : Returns shipments with the shipDate on or after the specified date.}
                            {--shipDateEnd= : Returns shipments with the shipDate on or after the specified date.}
                            {--voidDateStart= : Returns shipments voided on or after the specified date.}
                            {--voidDateEnd= : Returns shipments voided on or before the specified date.}
                            {--storeId= : Returns shipments whose orders belong to the specified store ID.}
                            {--includeShipmentItems : Specifies whether to include shipment items with results Default value: false.}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Get shipments from ShipStation';

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        $integration = Integration::with([])->where('name', Integration::NAME_SHIPSTATION)->firstOrFail();

        /** @var IntegrationInstance $integrationInstance */
        $integrationInstance = $integration->integrationInstances()->first();

        if (! $integrationInstance) {
            $this->error('There is no instance for '.$integration->name);

            return;
        }

        $job = new \App\Jobs\ShipStation\GetShipments($integrationInstance);
        if (empty($this->option('shipDateStart'))) {
            $this->input->setOption('shipDateStart', $this->calculateStartDate()->format('Y-m-d'));
        }
        $job->setRequestOptions(Arr::except(array_filter($this->options()), $this->getBuiltInOptions()));

        dispatch($job);

        $this->info('Added to the Queue, it will be process shortly');
    }

    private function calculateStartDate()
    {
        $threeDaysAgo = Carbon::now()->subDays(3)->startOfDay();
        $oldestAwaitingTracking = SalesOrderFulfillment::where('status', SalesOrderFulfillment::STATUS_SUBMITTED)
            ->where('created_at', '>=', $threeDaysAgo)
            ->oldest('created_at')
            ->where('fulfillment_type', SalesOrderFulfillment::TYPE_SHIPSTATION)
            ->value('created_at');
        if ($oldestAwaitingTracking) {
            $startDate = max($oldestAwaitingTracking, Carbon::now()->subDays(3)->startOfDay());
        } else {
            $startDate = Carbon::now()->subDays(3)->startOfDay();
        }

        return $startDate;
    }

    private function getBuiltInOptions()
    {
        return [
            'help',
            'quiet',
            'verbose',
            'version',
            'ansi',
            'no-ansi',
            'no-interaction',
            'env',
        ];
    }
}
