<?php

namespace App\Console\Commands;

use App\Models\SalesOrder;
use Facades\App\Services\Shopify\Orders\Actions\ShopifyDownloadOrder;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;

class RefreshClosedOrdersWithUnmappedLines extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:refresh-closed-orders-without-mappings';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Refreshes closed sku orders without mappings.';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $query = SalesOrder::with(['salesOrderLines'])
            ->where('order_status', SalesOrder::STATUS_CLOSED)
            ->whereHas('salesOrderLines', function (Builder $builder) {
                return $builder->where('is_product', 1)
                    ->whereNull('product_id');
            })
            ->whereHas('shopifyOrder'); // Only available for shopify orders now.

        $total = $query->count();
        $processed = 0;

        $query
            ->cursor()
            ->each(function (SalesOrder $salesOrder) use (&$processed, $total) {
                $this->output->info("Refreshing order: $salesOrder->sales_order_number.");

                // This is only available for shopify orders now.
                if (! $salesOrder->shopifyOrder) {
                    return false;
                }

                if ($salesOrder->order_status === SalesOrder::STATUS_CLOSED) {
                    $salesOrder->order_status = SalesOrder::STATUS_OPEN;
                    $salesOrder->save();
                }

                ShopifyDownloadOrderrefresh($salesOrder->shopifyOrder);

                $processed++;

                $this->output->info("Processed: $processed of $total.");

                return true;
            });

        $this->output->info("Refreshed $processed of $total orders.");

        return 0;
    }
}
