<?php

namespace App\Console\Commands;

use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Throwable;

class RecreateClosedOrdersWithBackorders extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:refresh-orders-with-backorders
                                    {--c : Run for closed orders. }
                                    {--s : Run for orders with stock on hand. }';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Attempts to recreate sales orders with backorder queues';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        set_time_limit(0);

        if ($this->option('c')) {
            $this->fixClosedOrders();
        }

        if ($this->option('s')) {
            $this->fixOnhandOrders();
        }

        return 0;
    }

    protected function fixOnhandOrders()
    {
        $this->output->info('Fixing orders with stock on hand.');

        $query = SalesOrderLine::with(['salesOrder', 'backorderQueue',
            'product.activeFifoLayers', 'salesOrder', ])
            ->whereHas('backorderQueue', function (Builder $builder) {
                return $builder->active();
            })
            ->whereHas('product.activeFifoLayers')
            ->whereHas('salesOrder', function (Builder $builder) {
                return $builder->where('fulfillment_status', SalesOrder::FULFILLMENT_STATUS_UNFULFILLED);
            });

        $processedOrders = [];

        $affected = $query->count();
        $skipped = 0;
        $processed = 0;

        $query->cursor()->each(function (SalesOrderLine $salesOrderLine) use (&$skipped, $affected, &$processed, &$processedOrders) {
            try {
                $salesOrder = $salesOrderLine->salesOrder;

                if (! $salesOrder) {
                    return;
                }

                if (in_array($salesOrder->id, $processedOrders)) {
                    return;
                }

                if (! $this->recreateOrder($salesOrder)) {
                    $skipped++;
                } else {
                    $processed++;
                }

                $this->output->info("$processed of $affected processed, recent: $salesOrder->sales_order_number");

                $processedOrders[] = $salesOrder->id;
            } catch (Throwable $exception) {
                Log::debug("Error fixing sales order: $salesOrder->sales_order_number", [
                    'message' => $exception->getMessage(),
                    'file' => $exception->getFile(),
                    'line' => $exception->getLine(),
                ]);
            }
        });

        $this->output->info("Completed $processed of $affected on-hand orders orders.");
    }

    protected function fixClosedOrders()
    {
        $this->output->info('Fixing closed orders.');

        $query = SalesOrder::with(['salesOrderLines', 'salesOrderLines.backorderQueue'])
            ->where(function (Builder $builder) {
                return $builder->whereNotNull('canceled_at')
                    ->orWhere('fulfillment_status', '=', SalesOrder::FULFILLMENT_STATUS_FULFILLED);
            })
            ->whereHas('salesOrderLines.backorderQueue', function (Builder $builder) {
                return $builder->active();
            });

        $affected = $query->count();
        $skipped = 0;
        $processed = 0;

        $query->cursor()->each(function (SalesOrder $salesOrder) use (&$skipped, $affected, &$processed) {
            if (! $this->recreateOrder($salesOrder)) {
                $skipped++;
            } else {
                $processed++;
            }

            $this->output->info("$processed of $affected processed.");
        });

        $this->output->info("Completed $processed of $affected closed orders.");
    }

    protected function recreateOrder(SalesOrder $salesOrder): bool
    {
        $salesChannelOrder = $salesOrder->order_document;
        if (! $salesChannelOrder) {
            $this->output->info("No sales channel order, skipping {$salesOrder->sales_order_number}");

            return false;
        }

        DB::beginTransaction();

        try {
            $salesOrder->delete();
            $salesChannelOrder->createSKUOrder();
            DB::commit();

            return true;
        } catch (Throwable $exception) {
            DB::rollBack();
            // We attempt to fix the sales order by line
            $this->fixLines($salesOrder->salesOrderLines()->whereHas('backorderQueue', function (Builder $builder) {
                return $builder->active();
            })->get());
            $this->output->info("Unable to refresh {$salesOrder->sales_order_number}, error: {$exception->getMessage()}");

            return false;
        }
    }

    protected function fixLines(Collection $salesOrderLines)
    {
        $salesOrderLines->each(function (SalesOrderLine $orderLine) {
            if ($orderLine->unfulfilled_quantity == 0 && $orderLine->canceled_quantity) {
                $orderLine->backorderQueue?->delete();
            }
        });
    }
}
