<?php

namespace App\Console\Commands\Patches;

use App\Data\InventoryLocationData;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class UpdateIntegrationInstanceInventorySchema extends Command
{

    protected $signature = 'sku:patch:update-integration-instance-inventory-schema';
    protected $description = 'Update the integration instance inventory schema.';

    public function handle(): int
    {
        $this->info('Updating integration instance inventory schema...');

        IntegrationInstance::query()
            ->whereHas('integration', function ($query) {
                $query->where('integration_type', Integration::TYPE_SALES_CHANNEL);
            })->each(function(IntegrationInstance $integrationInstance){

                DB::transaction(function() use ($integrationInstance){

                    $settings = $integrationInstance->integration_settings;
                    if(is_string($settings)){
                        $settings = json_decode($settings, true);
                    }

                    if(isset($settings['inventory'])) {
                        return;
                    }

                    $settings['inventory'] = [];

                    if(isset($settings['inventoryLocations'])){
                        $key = 'inventoryLocations';
                    }elseif(isset($settings['inventorySources'])){
                        $key = 'inventorySources';
                    } else {
                        $this->error('No inventory locations or sources found.');
                        return;
                    }

                    $settings['inventory']['locations'] = collect($settings[$key])
                        ->map(function($location){
                            return InventoryLocationData::from([
                                'id' => $location['id'] ?? ($location['shopify_location_id'] ?? null),
                                'name' => $location['name'] ?? ($location['shopify_location_name'] ?? null),
                                'selectedWarehouses' => collect($location['selectedWarehouses'] ?? [])
                                    ->pluck('id')->toArray(),
                                'masterOfStock' => $location['masterOfStock'] ?? null,
                                'inventoryModificationRules' => [
                                    'maxRuleType' => $location['maxRuleType'] ?? 'None',
                                    'maxRuleTypeValue' => $location['maxRuleTypeValue'] ?? null,
                                    'minRuleType' => $location['minRuleType'] ?? 'None',
                                    'minRuleTypeValue' => $location['minRuleTypeValue'] ?? null,
                                    'substractBufferStock' => $location['substractBufferStock'] ?? 0,
                                ],
                                'fulfillmentLatency' => $location['fulfillmentLatency'] ?? 0,
                                'lastSyncedAt' => $location['lastSyncedAt'] ?? null,
                            ]);
                        })->toArray();

                    $settings['inventory']['masterOfStock'] = IntegrationInstance::SETTINGS_OPTION_NAME_NEITHER;
                    $settings['inventory']['fulfillmentLatency'] = 0;
                    $settings['inventory']['lastSyncedAt'] = null;

                    unset($settings[$key]);

                    $integrationInstance->integration_settings = $settings;
                    $integrationInstance->save();

                });

            });

        $this->info('Integration instance inventory schema updated.');

        return 0;
    }

}