<?php

namespace App\Console\Commands\Inventory\Patches;

use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\ProductInventory;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class ReservePreAuditTrailInventory extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:inventory:patch:reserve-pre-audit-trail-inventory
                                {--ids= : Specify product ids }';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Make reservations for inventory that was fulfilled post audit trail but belonged to orders pre audit trail';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $ids = $this->option('ids') ? explode(',', $this->option('ids')) : false;

        /*
         * Check any products with negative reserved quantity
         */
        echo 'No audit trail lines with fulfillments:'."\n\n";

        $negative_reserved_products = ProductInventory::with([]);
        //->where('inventory_reserved', '<', 0)
        //->where('warehouse_id', 0);

        if ($ids) {
            $negative_reserved_products->whereIn('product_id', $ids);
        }

        $negative_reserved_products->get()
            ->each(function (ProductInventory $productInventory) {
                if ($productInventory->product->inventoryMovements->where('inventory_status', 'reserved')->sum('quantity') != $productInventory->inventory_reserved) {
                    $productInventory->inventory_reserved = $productInventory->product->inventoryMovements->where('inventory_status', 'reserved')->sum('quantity');
                    $productInventory->save();
                    echo 'Cache was invalid, fixing'."\n";

                    return;
                }
                DB::transaction(function () use ($productInventory) {
                    $product = $productInventory->product;
                    echo $product->sku."\n";
                    SalesOrderLine::with(['salesOrder'])
                        ->where('product_id', $product->id)
                        ->where('no_audit_trail', 1)
                        ->whereHas('salesOrderFulfillmentLines')
                        ->get()
                        ->take(1)
                        ->each(function (SalesOrderLine $salesOrderLine) use ($product) {
                            /*
                     * We'd want to
                     * a) Perform reservations (active -1, reserved +1) (Check Approval logic).  Date: sales order date?
                     * b) Remove no_audit_trail flag
                     */

                            $total_reserved_lines = $salesOrderLine->inventoryMovements->where('inventory_status', 'reserved')->sum('quantity');
                            $total_reserved_fulfillments = 0;
                            $salesOrderLine->salesOrderFulfillmentLines->each(function (SalesOrderFulfillmentLine $fulfillmentLine) use (&$total_reserved_fulfillments) {
                                $total_reserved_fulfillments += $fulfillmentLine->inventoryMovements->where('inventory_status', 'reserved')->sum('quantity');
                            });

                            if (($total_reserved_lines + $total_reserved_fulfillments) == 0) {
                                echo 'Sales order line reservations are ok, marking no audit trail false';
                                $salesOrderLine->no_audit_trail = 0;
                                $salesOrderLine->save();

                                return;
                            }

                            $audit_trail_start_date = $salesOrderLine->salesOrder->salesChannel->integrationInstance->audit_trail_start_date;

                            if ($salesOrderLine->salesOrder->salesChannel->integrationInstance->isShopify()) {
                                if ($pre_audit_trail_fulfillment_qty = collect($salesOrderLine->salesOrder->shopifyOrder['fulfillments'])->where('created_at', '<', $audit_trail_start_date)->pluck('line_items')->collapse()->sum('quantity')) {
                                    $pre_audit_trail_line = $salesOrderLine->replicate();
                                    $pre_audit_trail_line->split_from_line_id = $salesOrderLine->id;
                                    $pre_audit_trail_line->quantity = $pre_audit_trail_fulfillment_qty;
                                    $pre_audit_trail_line->save();

                                    $salesOrderLine->quantity -= $pre_audit_trail_fulfillment_qty;
                                    $salesOrderLine->save();
                                    $salesOrderLine->refresh();
                                }
                            }

                            $salesOrderLine->salesOrderFulfillmentLines->each(function (SalesOrderFulfillmentLine $fulfillmentLine) use ($salesOrderLine, $product) {
                                echo "\t".$salesOrderLine->salesOrder->sales_order_number."\n";

                                $fulfillmentLine->inventoryMovements()->delete();

                                $requiredQuantity = $fulfillmentLine->quantity;

                                if ($requiredQuantity == 0) {
                                    return;
                                }

                                $remainQuantity = $requiredQuantity;

                                do {
                                    /** @var FifoLayer $activeFifoLayer */
                                    $activeFifoLayer = $product->activeFifoLayers()->where('warehouse_id', $fulfillmentLine->salesOrderFulfillment->warehouse_id)->first();

                                    $inventoryMovement = new InventoryMovement();
                                    $inventoryMovement->inventory_movement_date = $fulfillmentLine->salesOrderFulfillment->fulfilled_at;
                                    $inventoryMovement->product_id = $product->id;
                                    $inventoryMovement->warehouse_id = $fulfillmentLine->salesOrderFulfillment->warehouse_id;
                                    $inventoryMovement->warehouse_location_id = $fulfillmentLine->warehouse->defaultLocation->id ?? null;

                                    //print 'remain (pre): ' . $remainQuantity . "\n";
                                    //print 'reqd (pre): ' . $requiredQuantity . "\n";
                                    if ($activeFifoLayer) {
                                        // if current fifo layer can't fulfill all required quantity.
                                        if ($remainQuantity > $activeFifoLayer->available_quantity) {
                                            $remainQuantity -= $activeFifoLayer->available_quantity;

                                            $requiredQuantity = $activeFifoLayer->available_quantity;
                                        } else {
                                            // remaining 0 if current fifo layer can fulfill all required quantity
                                            $requiredQuantity = $remainQuantity;
                                            $remainQuantity = 0;
                                        }

                                        // set quantity and fifo layer for inventory movement
                                        $inventoryMovement->quantity = -$requiredQuantity;
                                        $inventoryMovement->fifo_layer = $activeFifoLayer->id;

                                        $activeFifoLayer->fulfilled_quantity += $requiredQuantity;
                                        $activeFifoLayer->save();
                                        $activeFifoLayer->refresh();

                                        // add a link between sales order line and fifo layer
                                        $salesOrderLine->fifoLayers()->attach($activeFifoLayer->id, ['quantity' => $requiredQuantity]);
                                    } elseif ($requiredQuantity > 0) {
                                        echo 'It is impossible to have fulfilled this order'."\n";
                                        if ($this->confirm('Would you like to perform an adjustment on the date of the fulfillment?')) {
                                            $adjustment = new InventoryAdjustment([
                                                'adjustment_date' => $fulfillmentLine->salesOrderFulfillment->fulfilled_at,
                                                'product_id' => $product->id,
                                                'warehouse_id' => $fulfillmentLine->salesOrderFulfillment->warehouse_id,
                                                'quantity' => $requiredQuantity,
                                                'notes' => 'Adjustment to allow for historical shipment.',
                                                'unit_cost' => $product->average_cost ?? 0,
                                            ]);
                                            $adjustment->save();

                                            // Add fifo layer
                                            $fifoLayer = new FifoLayer();
                                            $fifoLayer->fifo_layer_date = $adjustment->adjustment_date;
                                            $fifoLayer->product_id = $adjustment->product_id;
                                            $fifoLayer->original_quantity = $adjustment->quantity;
                                            $fifoLayer->total_cost = $adjustment->quantity * $adjustment->unit_cost;
                                            $fifoLayer->warehouse_id = $adjustment->warehouse_id;
                                            $adjustment->fifoLayers()->save($fifoLayer);

                                            // Add inventory movement
                                            $adjustmentInventoryMovement = new InventoryMovement();

                                            $adjustmentInventoryMovement->inventory_movement_date = $adjustment->adjustment_date;
                                            $adjustmentInventoryMovement->product_id = $adjustment->product_id;
                                            $adjustmentInventoryMovement->quantity = $adjustment->quantity;
                                            $adjustmentInventoryMovement->type = InventoryMovement::TYPE_ADJUSTMENT;
                                            $adjustmentInventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
                                            $adjustmentInventoryMovement->warehouse_id = $adjustment->warehouse_id;
                                            $adjustmentInventoryMovement->fifo_layer = $fifoLayer->id;
                                            $adjustment->inventoryMovements()->save($adjustmentInventoryMovement);

                                            // set quantity and fifo layer for inventory movement
                                            $inventoryMovement->quantity = -$requiredQuantity;
                                            $inventoryMovement->fifo_layer = $fifoLayer->id;

                                            $fifoLayer->fulfilled_quantity += $requiredQuantity;
                                            $fifoLayer->save();
                                            $fifoLayer->refresh();

                                            $remainQuantity = 0;
                                        } else {
                                            return;
                                        }
                                    } else {
                                        $remainQuantity = 0;
                                    }

                                    //print 'remain (post): ' . $remainQuantity . "\n";
                                    //print 'reqd (post): ' . $requiredQuantity . "\n";
                                    // save movement with inventory status Active
                                    $inventoryMovement->type = InventoryMovement::TYPE_SALE;
                                    $inventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
                                    $salesOrderLine->inventoryMovements()->save($inventoryMovement);

                                    // save another movement with inventory status Reserved
                                    $reservedInventoryMovement = $inventoryMovement->replicate();
                                    $reservedInventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_RESERVED;
                                    $reservedInventoryMovement->quantity = $requiredQuantity;
                                    $salesOrderLine->inventoryMovements()->save($reservedInventoryMovement);

                                    $negativeReservedInventoryMovement = $inventoryMovement->replicate();
                                    $negativeReservedInventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_RESERVED;
                                    $negativeReservedInventoryMovement->quantity = -$requiredQuantity;
                                    $salesOrderLine->inventoryMovements()->save($negativeReservedInventoryMovement);
                                } while ($remainQuantity > 0);
                            });
                            $salesOrderLine->no_audit_trail = 0;
                            $salesOrderLine->save();

                            $salesOrderLine->refresh();
                            /*if (!($salesOrderLine->inventoryMovements->where('inventory_status', 'active')->sum('quantity') == -3 && $salesOrderLine->inventoryMovements->where('inventory_status', 'reserved')->sum('quantity') == 0 && $salesOrderLine->inventoryMovements->where('inventory_status', 'active')->count() == 2 && $salesOrderLine->inventoryMovements->where('inventory_status', 'reserved')->count() == 4))
                    {
                        print $salesOrderLine->inventoryMovements->where('inventory_status', 'active')->sum('quantity') . "\n";
                        print $salesOrderLine->inventoryMovements->where('inventory_status', 'reserved')->sum('quantity') . "\n";
                        $salesOrderLine->inventoryMovements->each(function(InventoryMovement $inventoryMovement) {
                            print $inventoryMovement->quantity . ' (' . $inventoryMovement->inventory_status . ') for ' . $inventoryMovement->fifo_layer->id . "\n";
                        });
                        print $salesOrderLine->inventoryMovements->where('inventory_status', 'active')->count() . "\n";
                        print $salesOrderLine->inventoryMovements->where('inventory_status', 'reserved')->count() . "\n";
                        print 'Tests failed';
                        exit;
                    }*/
                        });
                });
            });
    }
}
