<?php

namespace App\Console\Commands\Inventory\Patches;

use App\Exceptions\InsufficientStockException;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\BackorderQueue;
use App\Models\FifoLayer;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\Shopify\ShopifyOrder;
use App\Services\InventoryManagement\InventoryManager;
use App\Services\SalesOrder\FulfillSalesOrderService;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class HandlePostAuditTrailFulfillments extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:inventory:patch:handle-post-audit-trail-fulfillments
                            {--debug : Debug mode, ask for confirmations}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = '';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        Integration::with(['integrationInstances'])
            ->where('name', Integration::NAME_SHOPIFY)
            ->first()
            ->integrationInstances
            ->each(function (IntegrationInstance $instance) {
                $this->handleForIntegrationInstance($instance);
            });

        return 0;
    }

    protected function handleForIntegrationInstance(IntegrationInstance $instance)
    {
        $auditTrailStartDate = $instance->audit_trail_start_date;

        $results = DB::select('
SELECT so.id, so.requested_shipping_method, so.sales_order_number, sho.fulfillments from shopify_orders AS sho
INNER JOIN sales_orders AS so ON so.id = sho.sku_sales_order_id
LEFT JOIN sales_order_fulfillments AS sof ON sof.sales_order_id = so.id
INNER JOIN sales_order_lines AS sol ON sol.sales_order_id = so.id
where sol.product_id IS NOT NULL AND so.sales_order_number AND so.order_status = "closed" AND sof.id IS NULL AND date(sho.`created_at`) < "'.$auditTrailStartDate.'" and sho.`fulfillments` is not null and sho.`fulfillments` != "[]"
and (json_extract(sho.fulfillments, "$[0].created_at") >= "'.$auditTrailStartDate.'" and JSON_LENGTH(JSON_UNQUOTE(JSON_EXTRACT(sho.fulfillments, "$[*].id"))) = 1) GROUP BY so.id, so.requested_shipping_method, so.sales_order_number, sho.fulfillments;
');

        $this->info(count($results).' orders to fix');

        foreach ($results as $result) {
            DB::beginTransaction();
            try {
                $salesOrder = SalesOrder::find($result->id);
                $this->info($salesOrder->sales_order_number);

                $fulfillments = json_decode($result->fulfillments);

                if (count($fulfillments) != 1) {
                    throw new \Exception('Unexpected number of shopify fulfillments');
                }

                $fulfillment = $fulfillments[0];

                if ($fulfillment->status == 'cancelled') {
                    $this->info('Skipping due to cancelled fulfillment');
                    DB::rollBack();

                    continue;
                }

                /** @var SalesOrderLine $salesOrderLine */
                foreach ($salesOrder->salesOrderLines->whereNotNull('product_id') as $salesOrderLine) {
                    $movements = $salesOrderLine->inventoryMovements->where('type', 'sale')->groupBy('layer_id');

                    if ($movements->count()) {
                        $this->info('Removing existing movements for sales order line');

                        foreach ($movements as $layer_id => $layer_movements) {
                            foreach ($layer_movements as $movement) {
                                if ($movement->layer_type == BackorderQueue::class) {
                                    throw new \Exception('Backorder queue found');
                                }
                                $this->info($layer_id.': '.$movement->product->sku.', '.$movement->quantity.'('.$movement->inventory_status.')');
                            }
                        }
                        if ($this->option('debug') && ! $this->confirm('Would you like to delete these movements?')) {
                            echo 'OK, skipping'."\n";

                            continue;
                        }
                        foreach ($movements as $layer_id => $layer_movements) {
                            /*
                             * TODO: Need bright's help.  Deleting movements this way does not update products_inventory or fifo layers fulfilled qty cache
                             */
                            foreach ($layer_movements as $movement) {
                                $movement->delete();
                            }

                            $fifoLayer = FifoLayer::with([])->findOrFail($layer_id);
                            $fifoLayer->fulfilled_quantity = max(0, $fifoLayer->fulfilled_quantity - abs(collect($layer_movements)->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)->sum('quantity')));
                            $fifoLayer->save();
                        }
                        dispatch_sync(new UpdateProductsInventoryAndAvgCost([$salesOrderLine->product_id]));
                    }

                    $manager = new InventoryManager($salesOrderLine->warehouse_id, $salesOrderLine->product);

                    $applicableQty = $this->getFulfillmentLineById($salesOrderLine->sales_channel_line_id, $fulfillment)?->quantity;

                    if (! $applicableQty) {
                        $this->info("No applicable quantity for {$salesOrderLine->product->sku} ($salesOrderLine->sales_channel_line_id), skipping");

                        continue;
                    }

                    try {
                        $manager->takeFromStock($applicableQty, $salesOrderLine);
                    } catch (InsufficientStockException $ie) {
                        $this->handleLineAdjustments($salesOrderLine, $applicableQty, $fulfillment);
                        $manager->takeFromStock($applicableQty, $salesOrderLine);
                    }

                    $salesOrderLine->inventoryMovements()->update([
                        'inventory_movement_date' => Carbon::parse($fulfillment->created_at)->setTimezone('UTC'),
                    ]);
                }

                if ($this->option('debug') && ! $this->confirm('Would you like to create the fulfillment?')) {
                    DB::rollBack();
                }

                $salesOrder->open();

                $ssiLines = [];

                $warehouse_id = null;
                foreach ($fulfillment->line_items as $item) {
                    $matchingLine = $salesOrder->getMatchingUnfulfilledSalesOrderLineFromId($item->id);

                    /*
                     * For cases where shopify fulfilled it but sku considered it a non product line
                     */
                    if (! $matchingLine['product_id']) {
                        continue;
                    }

                    $warehouse_id = $matchingLine['warehouse_id'];
                    /*
                     * reset no audit trail for line
                     */
                    try {
                        $this->updateAuditTrailStatus($item, $salesOrder);
                    } catch (\Throwable $e) {
                        $this->info($e->getMessage());
                        DB::rollBack();

                        continue 2;
                    }

                    $ssiLines[] = [
                        'quantity' => $item->quantity,
                        'sales_order_line_id' => $matchingLine['id'],
                    ];
                }

                $request = [
                    'fulfilled_at' => Carbon::parse($fulfillment->created_at)->setTimezone('UTC'),
                    'fulfillment_lines' => $ssiLines,
                    'fulfillment_type' => 'manual',
                    'fulfillment_sequence' => 1,
                    'metadata' => json_encode(['signature_required' => false]),
                    'requested_shipping_method' => $result->requested_shipping_method ?? '',
                    'warehouse_id' => $warehouse_id,
                ];

                $fulfillment = FulfillSalesOrderService::make($salesOrder)->fulfillWithInputs($request, false, false);

                /*
                 * TODO: implement best practice of invalidating cache on write and recalculate after the heavy processing, and recalculating on first read as a backup
                 */
                dispatch_sync(new UpdateProductsInventoryAndAvgCost($salesOrder->salesOrderLines->whereNotNull('product_id')->pluck('product_id')->toArray()));

                if (is_array($fulfillment)) {
                    throw new \Exception($fulfillment[0]);
                }

                DB::commit();
            } catch (\Throwable $e) {
                DB::rollBack();
                dd($e->getMessage(), $e->getFile(), $e->getLine(), $e->getTraceAsString());
            }

            //tmp: exit after one iteration
        }
        exit;

        //TODO: If the fulfillment fails due to insufficient inventory, we need to get the most recent negative adjustment and prompt user to delete it, or skip the order

        /*
         *  $lastNegativeAdjustment = $salesOrderLine->product->inventoryMovements->where('warehouse_id', $salesOrderLine->warehouse_id)->where('type', 'adjustment')->where('quantity', '<', 0)->order('inventory_movement_date', 'DESC')->first();

        if ($lastNegativeAdjustment)
        {
            $adjustment = InventoryAdjustment::find($lastNegativeAdjustment->link_id);
            print 'A negative adjustment exists for ' . $lastNegativeAdjustment->product->sku . ' for ' . $lastNegativeAdjustment->quantity . ' (notes: ' . $adjustment->notes . ') on ' . $adjustment->adjustment_date . "\n";
            if (confirm('Would you like to delete this negative adjustment resulting in a positive inventory impact of ' . $adjustment->quantity . '?')) {

                 //TODO: Bright, does this adjustment deletion handle all the necessary movements and cache updates?

        $adjustment->delete();
    }
            else
            {
                print 'OK, then skipping sales order';
                DB::rollBack();
                continue 2;
            }
        }
         */

        /*
        $shopifyOrders = Order::with([])
            ->whereDate('created_at', '<', $auditTrailStartDate)
            ->whereNotNull('fulfillments')
            ->where('fulfillments', '!=', '[]')
            ->where('integration_instance_id', $instance->id);

        $shopifyOrders->where(function ($q) use ($auditTrailStartDate) {
            $q->whereRaw('json_extract(fulfillments, "$[0].created_at") >= "'.$auditTrailStartDate.'"');
            $q->whereRaw('JSON_LENGTH(JSON_UNQUOTE(JSON_EXTRACT(sho.fulfillments, "$[*].id"))) = 1');
        });
//->join('supplier_products', 'supplier_products.product_id', '=', 'products.id');
        $shopifyOrders->join('sales_orders AS so', 'so.'*/

        exit;

        /*
         * Case with multiple shopify fulfillments
         */

        $maxFulfillmentCount = 6;

        $shopifyOrders = ShopifyOrder::with([])
            ->whereDate('created_at', '<', $auditTrailStartDate)
            ->whereNotNull('fulfillments')
            ->where('fulfillments', '!=', '[]')
            ->where('integration_instance_id', $instance->id);

        $shopifyOrders->where(function ($q) use ($maxFulfillmentCount, $auditTrailStartDate) {
            $q->whereRaw('json_extract(fulfillments, "$[0].created_at") >= "'.$auditTrailStartDate.'"');

            for ($i = 1; $i < $maxFulfillmentCount; $i++) {
                $q->orWhereRaw('json_extract(fulfillments, "$['.$i.'].created_at") >= "'.$auditTrailStartDate.'"');
            }
        });

        //
        //            ->whereRaw('json_extract(fulfillments, "$[*].created_at") >= "' . $auditTrailStartDate . '"')
        //            ->whereDate('created_at', '<', $auditTrailStartDate);
        //

        /*
select * from `shopify_orders` AS sho
INNER JOIN sales_orders AS so ON so.id = sho.sku_sales_order_id
LEFT JOIN sales_order_fulfillments AS sof ON sof.sales_order_id = so.id
where sof.id IS NULL AND sho.integration_instance_id = ? AND date(sho.`created_at`) < ? and sho.`fulfillments` is not null and sho.`fulfillments` != ?
and (json_extract(sho.fulfillments, "$[0].created_at") >= "2021-11-02 00:00:00" or
json_extract(sho.fulfillments, "$[1].created_at") >= "2021-11-02 00:00:00" or
json_extract(sho.fulfillments, "$[2].created_at") >= "2021-11-02 00:00:00" or
json_extract(sho.fulfillments, "$[3].created_at") >= "2021-11-02 00:00:00" or
json_extract(sho.fulfillments, "$[4].created_at") >= "2021-11-02 00:00:00" or
json_extract(sho.fulfillments, "$[5].created_at") >= "2021-11-02 00:00:00")
         */

        /*
                $shopifyOrders->whereExists(function ($q) {
                    $q
                });
        */
        /*
        $shopifyOrders->each(function (Order $shopifyOrder) {
            print $shopifyOrder->salesOrder->salesOrderFulfillments->count();
            exit;
        });*/

        $this->warn('Total Orders Affected: '.$shopifyOrders->count().' for integration instance: '.$instance->name);
        dd($shopifyOrders->toSql(), $shopifyOrders->getBindings());
    }

    private function getFulfillmentLineById($id, $fulfillment)
    {
        $line = collect($fulfillment->line_items)
            ->where('id', $id)
            ->first();

        return $line;
    }

    private function updateAuditTrailStatus($line, $salesOrder)
    {
        $query = $salesOrder->salesOrderLines()->whereNotNull('product_id')->where('sales_channel_line_id', $line->id);

        if ($query->where('quantity', $line->quantity)->count() == 1) {
            $query->update([
                'no_audit_trail' => false,
            ]);
        } else {
            throw new \Exception('Split needed');
        }
    }

    private function handleLineAdjustments(SalesOrderLine $salesOrderLine, int $applicableQty, $fulfillment)
    {
        $product = $salesOrderLine->product;

        $applicableQty -= $product->activeFifoLayers->sum('available_quantity');

        if ($applicableQty <= 0) {
            return;
        }

        $adjustmentNumber = 0;

        do {
            // Take stock from most recent negative adjustment
            // for the same warehouse.
            /** @var InventoryAdjustment $adjustment */
            $adjustment = $product->inventoryAdjustments()
                ->where('quantity', '<', 0)
                ->where('warehouse_id', $salesOrderLine->warehouse_id)
                ->orderBy('adjustment_date', 'DESC')
                ->offset($adjustmentNumber)
                ->first();

            if (! $adjustment) {
                break;
            }

            if (abs($adjustment->quantity) <= $applicableQty) {
                // We delete the adjustment and
                // update the applicable qty.
                $message = "Delete Adjustment -> ID: $adjustment->id, Qty: $adjustment->quantity, Notes: $adjustment->notes?";
                if ($this->option('debug') && ! $this->confirm($message)) {
                    $adjustmentNumber++;

                    continue;
                }
                $adjustment->delete(false);
                $applicableQty -= abs($adjustment->quantity);
            } else {
                //throw new \Exception('Negative adjustment updated needed!!! Product: '.$product->sku);
                break;
            }
        } while ($applicableQty > 0);

        // If more quantity is needed, we create a positive adjustment
        if ($applicableQty > 0) {
            $message = "Create Adjustment -> Qty: $applicableQty for $product->sku?";
            if ($this->option('debug') && ! $this->confirm($message)) {
                return;
            }

            $adjustment = new InventoryAdjustment();
            $adjustment->quantity = $applicableQty;
            $adjustment->adjustment_date = $fulfillment->created_at;
            $adjustment->notes = 'sku.io integrity.';
            $adjustment->warehouse_id = $salesOrderLine->warehouse_id;
            $product->inventoryAdjustments()->save($adjustment);

            InventoryManager::with(
                $adjustment->warehouse_id,
                $adjustment->product
            )->addToStock(abs($adjustment->quantity), $adjustment, true, false);
        }
    }
}
