<?php

namespace App\Console\Commands\Inventory\Integrity;

use App\Console\Commands\Inventory\Integrity\Contracts\Identifier;
use App\Console\Commands\Inventory\Integrity\Contracts\Remedy;
use App\Models\InventoryMovement;
use App\Models\PurchaseOrderLine;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use Illuminate\Database\Eloquent\Builder;

class OutOfSyncInventoryMovementReferences extends Integrity implements Identifier, Remedy
{

    /**
     * @return Builder
     */
    public function outOfSyncInventoryMovementReferencesQuery(): Builder
    {
        $salesOrderLineQuery = InventoryMovement::query()
            ->selectRaw('
                inventory_movements.id,
                inventory_movements.reference movement_reference,
                sales_orders.sales_order_number link_reference
            ')
            ->join('sales_order_lines', function($join) {
                $join->on('sales_order_lines.id', '=', 'inventory_movements.link_id')
                    ->where('inventory_movements.link_type', SalesOrderLine::class);
            })
            ->join('sales_orders', 'sales_orders.id', '=', 'sales_order_lines.sales_order_id')
            ->where('inventory_movements.link_type', SalesOrderLine::class)
            ->whereColumn('sales_orders.sales_order_number', '!=', 'inventory_movements.reference');

        $salesOrderFulfillmentLineQuery = InventoryMovement::query()
            ->selectRaw("
                inventory_movements.id,
                inventory_movements.reference movement_reference,
                concat(sales_orders.sales_order_number, '.', sales_order_fulfillments.fulfillment_sequence) link_reference
            ")
            ->join('sales_order_fulfillment_lines', function($join) {
                $join->on('sales_order_fulfillment_lines.id', '=', 'inventory_movements.link_id')
                    ->where('inventory_movements.link_type', SalesOrderFulfillmentLine::class);
            })
            ->join('sales_order_fulfillments', 'sales_order_fulfillments.id', '=', 'sales_order_fulfillment_lines.sales_order_fulfillment_id')
            ->join('sales_orders', 'sales_orders.id', '=', 'sales_order_fulfillments.sales_order_id')
            ->where('inventory_movements.link_type', SalesOrderFulfillmentLine::class)
            ->whereRaw("concat(sales_orders.sales_order_number, '.', sales_order_fulfillments.fulfillment_sequence) != inventory_movements.reference");


        $purchaseOrderLineQuery = InventoryMovement::query()
            ->selectRaw('
                inventory_movements.id,
                inventory_movements.reference movement_reference,
                purchase_orders.purchase_order_number link_reference
            ')
            ->join('purchase_order_lines', function($join) {
                $join->on('purchase_order_lines.id', '=', 'inventory_movements.link_id')
                    ->where('inventory_movements.link_type', PurchaseOrderLine::class);
            })
            ->join('purchase_orders', 'purchase_orders.id', '=', 'purchase_order_lines.purchase_order_id')
            ->where('inventory_movements.link_type', PurchaseOrderLine::class)
            ->whereColumn('purchase_orders.purchase_order_number', '!=', 'inventory_movements.reference');

        return $salesOrderLineQuery->union($salesOrderFulfillmentLineQuery)->union($purchaseOrderLineQuery);
    }

    /**
     * @return void
     */
    public function identify(): void
    {
        $total = $this->outOfSyncInventoryMovementReferencesQuery()->count();
        $this->addMessage('There are '.$total.' inventory movements with references out of sync.', $total);
    }

    /**
     * @return void
     */
    public function examples(): void
    {
        $query = $this->outOfSyncInventoryMovementReferencesQuery();
        if ($query->count() == 0) {
            return;
        }

        $this->printMessage("EXAMPLES: Out of sync references\n");

        foreach ($query->take(5)->get() as $result) {
            $this->printMessage(
                "Inventory Movement ID: $result->id, movement reference: $result->movement_reference, 
                link reference: $result->link_reference");
        }
    }

    /**
     * @return string
     */
    public function description(): string
    {
        return 'Fix inventory movements with out of sync references.';
    }

    /**
     * @return void
     */
    public function remedy(): void
    {
        $results = $this->outOfSyncInventoryMovementReferencesQuery()->get();

        if($results->isEmpty()){
            $this->console->info('No out of sync references found.');
            return;
        }

        $data = [];
        foreach($results as $row){
            $data[] = [
                'id' => $row->id,
                'reference' => $row->link_reference
            ];
        }

        batch()->update(new InventoryMovement(), $data, 'id');

    }


}