<?php

namespace App\Console\Commands\Inventory\Integrity;

use App\Console\Commands\Inventory\Integrity\Contracts\Identifier;
use App\Console\Commands\Inventory\Integrity\Contracts\Remedy;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\InventoryMovement;
use App\Models\SalesOrderLine;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;

class OrphanedReservations extends Integrity implements Identifier, Remedy
{
    private function orphanedReservationsQuery(array $relations = []): Builder
    {
        return InventoryMovement::with($relations)
            ->select('inventory_movements.id', 'inventory_movements.link_id')
            ->leftJoin('sales_order_lines', 'sales_order_lines.id', 'inventory_movements.link_id')
            ->where('inventory_movements.link_type', SalesOrderLine::class)
            ->whereNull('sales_order_lines.id');
    }

    public function identify(): void
    {
        /**
         * Orphan reservations
         */
        $total = $this->orphanedReservationsQuery()->count();
        $this->addMessage('There are '.$total.' orphan reservation inventory movements', $total);
    }

    public function examples(): void
    {
        $query = $this->orphanedReservationsQuery(['product', 'link', 'link.salesOrder']);
        if ($query->count() == 0) {
            return;
        }

        $this->printMessage("EXAMPLES: Orphaned Reservations\n");

        foreach ($query->take(5)->get() as $result) {
            $this->printMessage("Inventory Movement ID: {$result->id} for sales order line id {$result->link_id}");
        }
    }

    public function remedy(): void
    {
        foreach ($this->orphanedReservationsQuery()->get() as $result) {
            DB::beginTransaction();
            try {
                /** @var InventoryMovement $inventoryMovement */
                $inventoryMovement = InventoryMovement::find($result->id);
                $product = $inventoryMovement->product;

                $this->console->info("$product->sku, $inventoryMovement->quantity ($inventoryMovement->inventory_status) belonging to FIFO layer $inventoryMovement->layer_id");

                if ($this->debugging() && ! $this->console->confirm('Would you like to delete this orphan inventory movement')) {
                    continue;
                }

                $inventoryMovement->delete();
                $inventoryMovement->reverseLayer();

                (new UpdateProductsInventoryAndAvgCost([$product->id]))->handle();

                DB::commit();
            } catch (\Throwable $e) {
                DB::rollBack();
                dd($e->getMessage(), $e->getFile(), $e->getLine(), $e->getTraceAsString());
            }
        }
    }

    public function description(): string
    {
        return 'Fix orphaned sales order line reservations.';
    }
}
