<?php

namespace App\Console\Commands\Inventory\Integrity;

use App\Console\Commands\Inventory\Integrity\Contracts\Identifier;
use App\Console\Commands\Inventory\Integrity\Contracts\Remedy;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\InventoryMovement;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use Illuminate\Support\Facades\DB;
use Throwable;

class MissingReservations extends Integrity implements Identifier, Remedy
{
    private function missingReservationsQuery(): \Illuminate\Database\Query\Builder
    {
        return DB::table('sales_order_lines', 'sol')
            ->select('sol.id')
            ->join('sales_orders', 'sales_orders.id', 'sol.sales_order_id')
            ->join('products', 'products.id', 'sol.product_id')
            ->where(function ($query) {
                $query->whereRaw("(SELECT sum(inventory_movements.quantity) FROM inventory_movements WHERE link_id = sol.id AND link_type = 'App\\\Models\\\SalesOrderLine' AND inventory_status = 'reserved' AND layer_type = 'App\\\Models\\\FifoLayer') != (SELECT -sum(im2.quantity) FROM sales_order_fulfillment_lines AS sofl
LEFT JOIN inventory_movements AS im2 ON im2.link_id = sofl.id AND im2.link_type = 'App\\\Models\\\SalesOrderFulfillmentLine'
WHERE sales_order_line_id = sol.id AND inventory_status = 'reserved')");
                $query->orWhere(function ($queryOr) {
                    $queryOr->whereRaw("(SELECT -sum(im2.quantity) FROM sales_order_fulfillment_lines AS sofl
LEFT JOIN inventory_movements AS im2 ON im2.link_id = sofl.id AND im2.link_type = 'App\\\Models\\\SalesOrderFulfillmentLine'
WHERE sales_order_line_id = sol.id AND inventory_status = 'reserved') IS NOT NULL");
                    $queryOr->whereRaw("(SELECT sum(inventory_movements.quantity) FROM inventory_movements WHERE link_id = sol.id AND link_type = 'App\\\Models\\\SalesOrderLine' AND inventory_status = 'reserved' AND layer_type = 'App\\\Models\\\FifoLayer') IS NULL");
                });
                $query->orWhere(function ($queryOr) {
                    $queryOr->whereRaw("(SELECT -sum(im2.quantity) FROM sales_order_fulfillment_lines AS sofl
LEFT JOIN inventory_movements AS im2 ON im2.link_id = sofl.id AND im2.link_type = 'App\\\Models\\\SalesOrderFulfillmentLine'
WHERE sales_order_line_id = sol.id AND inventory_status = 'reserved') IS NULL");
                    $queryOr->whereRaw("(SELECT sum(inventory_movements.quantity) FROM inventory_movements WHERE link_id = sol.id AND link_type = 'App\\\Models\\\SalesOrderLine' AND inventory_status = 'reserved' AND layer_type = 'App\\\Models\\\FifoLayer') IS NOT NULL");
                });
            })
            ->where(function ($query) {
                $query->where('sales_orders.is_fba', false);
                $query->orWhereNull('sales_orders.is_fba');
            })
            ->where('sales_orders.fulfillment_status', '!=', SalesOrder::FULFILLMENT_STATUS_OVER_FULFILLED)
            ->where('sales_orders.order_status', SalesOrder::STATUS_CLOSED);
    }

    public function identify(): void
    {
        $total = $this->missingReservationsQuery()->count();
        $this->addMessage('There are '.$total.' sales order lines with missing reservations issues', $total);
    }

    public function examples(): void
    {
        $query = $this->missingReservationsQuery();
        if ($query->count() == 0) {
            return;
        }

        $this->printMessage("EXAMPLES: Missing Reservations\n");

        foreach ($query->take(5)->get() as $result) {
            /** @var SalesOrderLine $salesOrderLine */
            $salesOrderLine = SalesOrderLine::with(['product', 'salesOrder'])->findOrFail($result->id);
            $this->printMessage("{$salesOrderLine->salesOrder->sales_order_number} (SKU: {$salesOrderLine->product->sku})");
        }
    }

    public function remedy(): void
    {
        foreach ($this->missingReservationsQuery()->get() as $result) {
            DB::beginTransaction();
            try {
                /** @var SalesOrderLine $salesOrderLine */
                $salesOrderLine = SalesOrderLine::with(['product', 'salesOrder', 'inventoryMovements'])->findOrFail($result->id);
                $product = $salesOrderLine->product;
                $salesOrder = $salesOrderLine->salesOrder;

                $this->console->info("Sales Order ($salesOrder->order_status) ".$salesOrder->sales_order_number.', SKU: '.$product->sku.' (Qty: '.$salesOrderLine->quantity.') Line ID ('.$salesOrderLine->id.')');

                $totalReserved = $salesOrderLine->inventoryMovements
                    ->where('inventory_status', 'reserved')
                    ->sum('quantity');

                if ($totalReserved == $salesOrderLine->quantity) {
                    $this->console->info('Sales order line appears to have correct reservations for sales order line');
                }

                if ($salesOrderLine->salesOrderFulfillmentLines->count() == 0) {
                    $this->console->info('Sales order is closed, but has the line has reservations and no fulfillments');

                    if ($this->debugging() && ! $this->console->confirm('Would you like to delete the movements for the line?')) {
                        continue;
                    }

                    $movements = $salesOrderLine->inventoryMovements->where('type', 'sale')->where('link_type', SalesOrderLine::class)->groupBy('layer_id');

                    foreach ($movements as $layer_movements) {
                        /** @var InventoryMovement $movement */
                        foreach ($layer_movements as $movement) {
                            $this->console->info('deleting inventory movement '.$movement->id);
                            $movement->delete();
                            $movement->reverseLayer();
                        }
                    }
                }

                foreach ($salesOrderLine->salesOrderFulfillmentLines as $salesOrderFulfillmentLine) {
                    if (-$salesOrderFulfillmentLine->inventoryMovements->sum('quantity') != $salesOrderFulfillmentLine->quantity) {
                        /*
                         * At this point I think we can delete existing movements for sales order fulfillment line and recreate using reverseReservedInventoryMovements
                         */
                        $this->console->info('Sales order line appears to have fulfillment movements different from the line quantity.');

                        if ($this->debugging() && ! $this->console->confirm('Would you like to delete and recreate the fulfillment movements?')) {
                            continue;
                        }

                        $salesOrderFulfillmentLine->inventoryMovements()->delete();
                        $salesOrderFulfillmentLine->negateReservationMovements();

                        (new UpdateProductsInventoryAndAvgCost([$salesOrderLine->product_id]))->handle();
                    }
                }
                DB::commit();
            } catch (Throwable $e) {
                DB::rollBack();
                dd($e->getMessage(), $e->getFile(), $e->getLine(), $e->getTraceAsString());
            }
        }
    }

    public function description(): string
    {
        return 'Fix sales order lines with missing reservations';
    }
}
