<?php

namespace App\Console\Commands\Inventory\Integrity;

use App\Console\Commands\Inventory\Integrity\Contracts\Identifier;
use App\Console\Commands\Inventory\Integrity\Contracts\Remedy;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use Illuminate\Support\Facades\DB;

class InvalidFulfilledQuantityCache extends Integrity implements Identifier, Remedy
{
    public function identify(): void
    {
        $query = $this->getFulfilledQuantityCacheMismatchQuery();
        $total = $query->count();
        $this->addMessage('There are '.$total.' products with invalid fulfilled quantity cache.', $total);
    }

    public function examples(): void
    {
        $query = $this->getFulfilledQuantityCacheMismatchQuery();
        $this->printMessage("EXAMPLES - Invalid Fulfilled Quantity Cache:\n");

        /** @var SalesOrderLine $fulfilled */
        foreach ($query->take(5)->get() as $fulfilled) {
            /** @var Product $product */
            $product = Product::query()->findOrFail($fulfilled->product_id);
            /** @var SalesOrder $salesOrder */
            $salesOrder = SalesOrder::query()->findOrFail($fulfilled->sales_order_id);
            $this->printMessage("SO: $salesOrder->sales_order_number ($salesOrder->id), SKU: $product->sku, Fulfilled Qty: $fulfilled->fulfilled_quantity, Fulfilled Cache: $fulfilled->cached_fulfilled_quantity");
        }
    }

    public function description(): string
    {
        return 'Re-calculate the fulfilled quantity cache';
    }

    public function remedy(): void
    {
        $this->getFulfilledQuantityCacheMismatchQuery()->get()->each(function ($fulfilled) {
            /** @var SalesOrderLine $salesOrderLine */
            $salesOrderLine = SalesOrderLine::query()->findOrFail($fulfilled->id);
            $salesOrderLine->fulfilled_quantity = $fulfilled->fulfilled_quantity;
            $salesOrderLine->update();
        });
    }

    private function getFulfilledQuantityCacheMismatchQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return SalesOrderLine::select(
            'sol.id',
            'sol.sales_order_id',
            'sol.product_id',
            'sol.fulfilled_quantity as cached_fulfilled_quantity',
            DB::raw('COALESCE(SUM(sofl.quantity), 0) as fulfilled_quantity')
        )
            ->from('sales_order_lines as sol')
            ->leftJoin('sales_order_fulfillment_lines as sofl', function ($join) {
                $join->on('sofl.sales_order_line_id', '=', 'sol.id');
            })
            ->groupBy('sol.id', 'sol.sales_order_id', 'sol.product_id', 'sol.fulfilled_quantity')
            ->havingRaw('sol.fulfilled_quantity != COALESCE(SUM(sofl.quantity), 0)');
    }
}
