<?php

namespace App\Console\Commands\Inventory\Integrity;

use App\Exceptions\InsufficientStockException;
use App\Models\InventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\SalesChannel;
use App\Models\SalesCredit;
use App\Models\SalesCreditLine;
use App\Models\SalesCreditReturnLine;
use App\Models\SalesOrder;
use App\Models\Shopify\ShopifyOrder;
use App\Models\Shopify\ShopifyOrderMapping;
use App\Repositories\Shopify\ShopifyOrderMappingRepository;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class CleanupExtraSalesCreditsCommand extends Command
{
    private ShopifyOrderMappingRepository $shopifyOrderMappingRepository;

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:shopify:orders:patch:cleanup-extra-sales-credits
                               {--debug : confirm mapping and deleting}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Clean up extra sales credits';

    public function __construct()
    {
        parent::__construct();

        $this->shopifyOrderMappingRepository = app(ShopifyOrderMappingRepository::class);
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('Syncing Shopify Mappings...');
        $this->syncShopifyMappings();

        $this->info('Syncing Sales Credits...');
        if (! $this->syncRefunds()) {
            return self::SUCCESS;
        }

        $this->info('Deleting Extra Sales Credits...');
        $this->deleteExtraSalesCredits();

        $this->info('Done');

        return self::SUCCESS;
    }

    /**
     * Add `line_type` to existing shopify mappings
     */
    private function syncShopifyMappings(): void
    {
        $this->shopifyOrderMappingRepository->fixLineTypeForFulfillments();
        $this->shopifyOrderMappingRepository->fixLineTypeAndLinkTypeForRefundAdjustments();
        $this->shopifyOrderMappingRepository->remapRefundMappingsForCancellations();
        $this->shopifyOrderMappingRepository->remapCancellationMappings();
        $this->shopifyOrderMappingRepository->remapRefundMappingsForSalesCreditLineRefunds();
        $this->shopifyOrderMappingRepository->remapRefundMappingsForSalesCreditLineRefundAdjustments();
        //TODO: now we can add a unique key on the `shopify_order_mappings` table
    }

    /**
     * Map Shopify refunds to SKU sales credits and handle missing refunds
     */
    private function syncRefunds(): bool
    {
        $debug = $this->option('debug');

        $query = self::unprocessedRefundsQuery();

        if (! $count = $query->count()) {
            $this->info('There are no Shopify refunds to sync');

            return true;
        }

        if (! $this->confirm("There are {$count} Shopify refunds to sync, sure?")) {
            return false;
        }

        $mapped = $query->eachById(function (ShopifyOrder $order) use (&$debug) {
            $fulfilledLineItemIds = collect($order->fulfillments)
                ->where('status', 'success')
                ->pluck('line_items')
                ->flatten(1)
                ->pluck('id')
                ->unique();

            $needToHandleRefunds = false;
            // trying to map Shopify refunds with SKU sales credits
            foreach ($order->refunds as $refund) {
                // empty lines and adjustments, skip it
                if (empty($refund['refund_line_items']) && empty($refund['order_adjustments'])) {
                    continue;
                }
                // already processed
                if ($order->isRefundProcessed($refund['id'])) {
                    continue;
                }

                if ($debug) {
                    $choice = $this->choice(
                        "{$order->name}, Do you want to map refund ID: {$refund['id']}?",
                        ['Yes', 'Skip', 'Cancel', 'Yes All'],
                        0
                    );
                    // Skip: don't map the refund
                    if ($choice == 'Skip') {
                        continue;
                    }
                    // Cancel: cancel the operation(mapping and deleting)
                    if ($choice == 'Cancel') {
                        return false;
                    }
                    // Yes All: stop debugging mode
                    if ($choice == 'Yes All') {
                        $debug = false;
                    }
                }

                $this->info("mapping {$order->name}, refund ID: {$refund['id']}");

                $credits = collect($refund['refund_line_items'])
                    ->whereIn('restock_type', ['return', 'no_restock'])
                    ->whereIn('line_item_id', $fulfilledLineItemIds);

                $needToHandleRefunds = true;
            }
            // if there are still unmapped refunds, call handleRefunds function
            if ($needToHandleRefunds) {
                $order->handleRefunds();
            }
        });

        return $mapped !== false;
    }

    public static function unprocessedRefundsQuery(): Builder
    {
        return ShopifyOrder::query()
            ->whereRelation('salesOrder', 'order_status', '!=', SalesOrder::STATUS_DRAFT)
            ->whereHas('orderMappings', fn ($q) => $q->where('link_type', ShopifyOrderMapping::LINK_TYPE_REFUNDS), '<', DB::raw('`refund_lines_count`'));
    }

    private function deleteExtraSalesCredits()
    {
        $debug = $this->option('debug');

        $shopifySalesChannels = SalesChannel::query()->whereHas('integrationInstance', fn ($q) => $q->shopify())->get();

        $query = SalesCreditLine::query()
            ->join('sales_credits', 'sales_credit_lines.sales_credit_id', '=', 'sales_credits.id')
            ->join('sales_orders', function (JoinClause $join) use ($shopifySalesChannels) {
                $join->on('sales_credits.sales_order_id', '=', 'sales_orders.id')
                    ->where('sales_orders.order_status', '!=', SalesOrder::STATUS_DRAFT)
                    ->whereIn('sales_channel_id', $shopifySalesChannels->pluck('id'));
            })
            ->leftJoin('shopify_order_mappings', function (JoinClause $join) {
                $join->on('sku_link_id', '=', 'sales_credit_lines.id')
                    ->where('sku_link_type', SalesCreditLine::class);
            })
            ->whereNull('shopify_order_mappings.id');
        if (! $count = $query->count()) {
            $this->info('There are no extra sales credits');

            return;
        }

        if (! $this->confirm("There are {$count} extra sales credits that will be deleted, sure?")) {
            return;
        }

        // delete extra sales credits
        $query->select('sales_credit_lines.*')
            ->eachById(function (SalesCreditLine $salesCreditLine) use (&$debug) {
                $salesCredit = $salesCreditLine->salesCredit;

                if ($debug) {
                    $choice = $this->choice(
                        "Do you want to delete sales credit line  (ID: {$salesCreditLine->id}), Sales Order: {$salesCredit->salesOrder->sales_order_number}?",
                        ['Yes', 'Skip', 'Cancel', 'Yes All'],
                        0
                    );
                    // Skip: don't map the refund
                    if ($choice == 'Skip') {
                        return; // continue
                    }
                    // Cancel: cancel the operation(deleting)
                    if ($choice == 'Cancel') {
                        return false;
                    }
                    // Yes All: stop debugging mode
                    if ($choice == 'Yes All') {
                        $debug = false;
                    }
                }

                $this->info("Deleting Sales Credit line (ID: {$salesCreditLine->id}), Sales Order: ".(($salesCredit->salesOrder->sales_order_number) ?? 'N/A'));
                try {
                    $salesCreditLine->delete();
                    if ($salesCredit->salesCreditLines->isEmpty()) {
                        $salesCredit->delete();
                    }
                } catch (InsufficientStockException $insufficientStockException) {
                    $this->warn("InsufficientStock, Can't delete sales credit {$salesCreditLine->salesCredit->sales_credit_number} (ID: {$salesCreditLine->salesCredit->id}), Sales Order: {$salesCreditLine->salesCredit->salesOrder->sales_order_number}");
                    if ($debug && ! $this->confirm('We will create new inventory adjustments rather than the sales credit returns, sure?')) {
                        return; // skip
                    }
                    // delete return lines to detect which one that thrown an InsufficientStockException
                    $salesCreditLine->salesCredit->salesCreditReturnsLines->each(function (SalesCreditReturnLine $creditReturnLine) use ($salesCredit) {
                        try {
                            $creditReturnLine->delete();
                        } catch (InsufficientStockException $insufficientStockException) {
                            DB::transaction(function () use ($salesCredit, $creditReturnLine) {
                                /**
                                 * we just link the existing fifo layer and inventory movement with a new inventory adjustment
                                 * to cover the existing movements only
                                 */
                                $returnInventoryMovement = $creditReturnLine->getOriginatingMovement();
                                // a new adjustment from return movement data
                                $adjustment = new InventoryAdjustment($returnInventoryMovement->getAttributes());
                                $adjustment->adjustment_date = $returnInventoryMovement->inventory_movement_date;
                                $adjustment->notes = 'sku.io integrity, to remove extra sales credit';
                                $adjustment->link_id = $salesCredit->salesOrder->id;
                                $adjustment->link_type = SalesOrder::class;
                                $adjustment->save();
                                // link return fifo layer to the adjustment
                                $adjustment->fifoLayers()->save($creditReturnLine->getFifoLayer());
                                // link return movement to the adjustment
                                $returnInventoryMovement->type = InventoryMovement::TYPE_ADJUSTMENT;
                                $adjustment->inventoryMovements()->save($returnInventoryMovement);
                                // try to delete the return line again
                                $creditReturnLine->delete();
                            });
                        }
                    });
                    // delete the sales credit
                    if ($salesCredit->isEmpty()) {
                        $salesCredit->delete();
                    }
                }
            }, 100, 'sales_credit_lines.id', 'id');
    }

    /**
     * Trying to map Shopify refund to a SKU sales credit
     */
    private function mapShopifyRefund(ShopifyOrder $order, array $refund, Collection $credits): bool
    {
        $mappedSalesCreditIds = $this->shopifyOrderMappingRepository->getSalesCreditIdsForShopifyOrder($order);

        $unmappedSalesCredits = $order->salesOrder->salesCredits->whereNotIn('id', $mappedSalesCreditIds);

        // map by lines
        $salesCredit = $this->getSalesCreditByLines($refund, $unmappedSalesCredits, $credits);
        if ($salesCredit) {
            $order->markRefundAsProcessed($refund['id'], $salesCredit->id, SalesCredit::class);

            return true;
        }

        return false;
    }

    /**
     * Get SKU sales credit that has same Shopify refund lines
     */
    private function getSalesCreditByLines(array $refund, Collection $unmappedSalesCredits, Collection $credits): ?SalesCredit
    {
        $adjustLines = collect($refund['order_adjustments'])->map(fn ($adjustLine) => ['quantity' => 1, 'line_item_id' => $adjustLine['reason']]);
        $shopifyLineItems = $credits->merge($adjustLines)->pluck('quantity', 'line_item_id')->toArray();

        return $unmappedSalesCredits->filter(function (SalesCredit $salesCredit) use ($shopifyLineItems) {
            $skuLines = $salesCredit->salesCreditLines
                ->map(fn (SalesCreditLine $creditLine) => ['quantity' => $creditLine->quantity, 'line_item_id' => $creditLine->salesOrderLine->sales_channel_line_id ?? $creditLine->description])
                ->pluck('quantity', 'line_item_id')->toArray();

            $skuLinesCount = count($skuLines);

            // same lines and quantities
            if (count($shopifyLineItems) == $skuLinesCount && count(array_intersect_assoc($shopifyLineItems, $skuLines)) == $skuLinesCount) {
                return true;
            }

            return false;
        })->first();
    }
}
