<?php

namespace App\Console\Commands\Inventory\Health\Fix\Traits;

use App\Models\InventoryMovement;
use Illuminate\Support\Collection;
use function Laravel\Prompts\confirm;
use function Laravel\Prompts\search;
use function Laravel\Prompts\select;
use function Laravel\Prompts\table;

trait FixInventoryMovementsTrait
{
    public function showProblemRecords(Collection $data): void
    {
        if (confirm("Show problem records?")) {
            $headers = [
                'ID',
                'Reference',
                'Status',
                'Product ID',
                'SKU',
                'Qty',
                'Warehouse',
            ];

            $rows = $data->map(function(InventoryMovement $row) {
                return [
                    $row->id,
                    $row->reference,
                    $row->inventory_status,
                    $row->product_id,
                    $row->product->sku,
                    $row->quantity,
                    $row->warehouse->name,
                ];
            })->toArray();

            table(
                headers: $headers,
                rows: $rows
            );

            $this->sendCsv($headers, $rows);
        }
    }

    protected function filter(): void
    {
        $fixSelection = select(
            label: "Which Inventory Movements would you like to fix?",
            options: [
                'all' => 'All',
                'reference' => 'Specific reference',
                'product' => 'Specific product',
            ],
        );

        if ($fixSelection === 'reference') {
            $reference = search(
                label: "Search for a reference",
                options: fn(string $value) => $this->data->filter(
                    fn($movement) => stripos($movement->reference, $value) !== false
                )->pluck('reference', 'id')->all(),
            );
            $this->data = $this->data->where('reference', $reference);
        } elseif ($fixSelection === 'product') {
            $productId = search(
                label: "Search for a product SKU",
                options: fn(string $value) => $this->data->filter(
                    fn($movement) => stripos($movement->product->sku, $value) !== false
                )->pluck('product.sku', 'id')->all(),
            );
            $this->data = $this->data->where('product_id', $productId);
        }
    }
}
