<?php

namespace App\Console\Commands\Inventory\Health\Fix;

use App\Console\Commands\Inventory\Health\AbstractInventoryHealthCommandHelper;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\StockTake;
use App\Repositories\FifoLayerRepository;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Throwable;
use function Laravel\Prompts\confirm;
use function Laravel\Prompts\info;
use function Laravel\Prompts\select;
use function Laravel\Prompts\search;
use function Laravel\Prompts\table;
use function Laravel\Prompts\text;

class FixOversubscribedFifoLayersCommandHelper extends AbstractInventoryHealthCommandHelper
{
    protected FifoLayerRepository $layers;
    protected Collection $fifoLayersToZeroOut;

    public function __construct(Command $console)
    {
        parent::__construct($console);
        $this->layers = app(FifoLayerRepository::class);
        $this->fifoLayersToZeroOut = collect();
    }

    public function getDescription(): string
    {
        return 'FIFO layers that are oversubscribed';
    }

    public function getQuery(): Builder
    {
        return $this->health->overSubscribedFifoLayersQuery();
    }

    /** @var FifoLayer $dataItem */
    protected function getFixLabel(mixed $dataItem): string
    {
        return "Fixing for FIFO Layer $dataItem->id with SKU {$dataItem->product->sku} for overage " . -$dataItem->inventoryMovements->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)->sum('quantity');
    }

    /**
     * @throws Throwable
     * @var FifoLayer $dataItem
     */
    protected function fix($dataItem): void
    {
        $remainingOverSubscribedQuantity = $this->healthManager->fixOverSubscribedFifoLayer($dataItem);
        if ($remainingOverSubscribedQuantity > 0) {
            $this->warnings->add("Could not fix all overages for FIFO Layer $dataItem->id ({$dataItem->product->sku} at {$dataItem->warehouse->name}).  Remaining overage is $remainingOverSubscribedQuantity");
            $this->fifoLayersToZeroOut->add($dataItem);
        }
    }

    protected function postFix(): bool
    {
        if ($this->fifoLayersToZeroOut->isNotEmpty())
        {
            if (confirm('Would you like to create a stock take for the remaining overages so that the Oversubscribed FIFO fix can be ran again?')) {
                $notes = text('Please enter notes for the stock take');
                $stockTakesCreated = $this->healthManager->createStockTakesForOverages($this->fifoLayersToZeroOut, $notes);
                if ($stockTakesCreated->isEmpty()) {
                    info('No stock takes were created');
                    return false;
                } else {
                    $stockTakesCreated->each(function (StockTake $stockTake) {
                        info("Stock take created (ID: $stockTake->id) for warehouse {$stockTake->warehouse->name}");
                    });
                    info("Retrying FIFO Layer over-subscription fixes...");
                    $this->fifoLayersToZeroOut = collect();
                    return true;
                }
            }
        }
        return false;
    }

    public function showProblemRecords(Collection $data): void
    {
        if (confirm("Show problem records?")) {
            $headers = [
                'ID',
                'FIFO Date',
                'Product',
                'Warehouse',
                'Available Quantity (Cached)',
                'Available Quantity (Calculated)',
            ];

            $rows = $data->map(function(FifoLayer $row) {
                return [
                    $row->id,
                    $row->fifo_layer_date,
                    $row->product->sku . " (ID: $row->product_id)",
                    $row->warehouse->name . " (ID: $row->warehouse_id)",
                    $row->available_quantity,
                    $row->inventoryMovements->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)->sum('quantity'),
                ];
            })->toArray();

            table(
                headers: $headers,
                rows: $rows
            );

            $this->sendCsv($headers, $rows);
        }
    }

    protected function filter(): void
    {
        $fixSelection = select(
            label: "Which FIFO layers would you like to fix?",
            options: [
                'all' => 'All',
                'fifo' => 'Specific FIFO Layer',
                'product' => 'Specific Product',
                'warehouse' => 'Specific Warehouse',
            ],
        );

        if ($fixSelection == 'fifo') {
            $fifoLayerId = text("Please enter the ID of the FIFO layer you would like to fix");
            $this->data = $this->data->where('id', $fifoLayerId);
        } elseif ($fixSelection == 'product') {
            $productId = search(
                label: "Search for a product SKU",
                options: fn(string $value) => $this->data->filter(
                    fn($fifoLayer) => stripos($fifoLayer->product->sku, $value) !== false
                )->pluck('product.sku', 'product_id')->all(),
            );
            $this->data = $this->data->where('product_id', $productId);
        } elseif ($fixSelection == 'warehouse') {
            $warehouseId = search(
                label: "Search for a warehouse",
                options: fn(string $value) => $this->data->filter(
                    fn($fifoLayer) => stripos($fifoLayer->warehouse->name, $value) !== false
                )->pluck('warehouse.name', 'warehouse_id')->all(),
            );
            $this->data = $this->data->where('warehouse_id', $warehouseId);
        }

        if (confirm(
            label: "Would you like recalculate the FIFO layer cache first? (Takes awhile)",
            default: false
        )) {
            customlog('inventory-fixes', 'Recalculating cache for all FIFO layers', days: null);
            FifoLayer::each(function (FifoLayer $fifoLayer) {
                $this->layers->validateFifoLayerCache($fifoLayer);
            });
        }
    }
}