<?php

namespace App\Console\Commands\Inventory\Health\Fix;

use App\Console\Commands\Inventory\Health\AbstractInventoryHealthCommandHelper;
use App\Console\Commands\Inventory\Health\Fix\Traits\FixSalesOrderLinesTrait;
use App\Data\ProductWarehouseQuantityData;
use App\Models\InventoryMovement;
use App\Models\SalesOrderLine;
use App\Models\StockTake;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Throwable;
use function Laravel\Prompts\confirm;
use function Laravel\Prompts\info;
use function Laravel\Prompts\text;

class FixExternallyFulfilledSalesOrderLinesWithMovementsCommandHelper extends AbstractInventoryHealthCommandHelper
{
    use FixSalesOrderLinesTrait;

    protected Collection $productWarehouseQuantities;

    public function __construct(Command $console)
    {
        parent::__construct($console);
        $this->productWarehouseQuantities = collect();
    }

    public function getDescription(): string
    {
        return 'Externally fulfilled sales order lines with inventory movements';
    }

    public function getQuery(): Builder
    {
        return $this->health->externallyFulfilledSaleOrderLinesWithInventoryMovementsQuery();
    }

    /** @var SalesOrderLine $dataItem */
    protected function getFixLabel(mixed $dataItem): string
    {
        return "Fixing for sales order {$dataItem->salesOrder->sales_order_number} line $dataItem->id with SKU {$dataItem->product->sku} for quantity $dataItem->quantity";
    }

    /**
     * @throws Throwable
     * @var SalesOrderLine $dataItem
     */
    protected function fix($dataItem): void
    {
        $deletedMovements = $this->healthManager->deleteSalesOrderReservation($dataItem);
        if ($deletedMovements->isNotEmpty()) {
            $deletedActiveMovements = $deletedMovements
                ->where('warehouse_id', $dataItem->warehouse_id)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE);
            $this->productWarehouseQuantities->add(ProductWarehouseQuantityData::from([
                'product_id' => $dataItem->product_id,
                'warehouse_id' => $dataItem->warehouse_id,
                'quantity' => $deletedActiveMovements->sum('quantity')
            ]));
        }
    }

    protected function postFix(): bool
    {
        // Not sure if it makes sense to allow the negative adjustment.
        return false;
        if ($this->productWarehouseQuantities->isNotEmpty())
        {
            if (confirm('Would you like to create a negative stock take for the externally fulfilled movement lines deleted to create a net zero effect on inventory?')) {
                $notes = text('Please enter notes for the stock take');
                $stockTakesCreated = $this->healthManager->createStockTakeForProductWarehouseQuantities($this->productWarehouseQuantities, $notes, asAdjustment: true);
                if ($stockTakesCreated->isEmpty()) {
                    info('No stock takes were created');
                } else {
                    $stockTakesCreated->each(function (StockTake $stockTake) {
                        info("Stock take created (ID: $stockTake->id) for warehouse {$stockTake->warehouse->name}");
                    });
                    $this->productWarehouseQuantities = collect();
                }
                return false;
            }
        }
        return false;
    }
}