<?php

namespace App\Console\Commands;

use App\Models\BackorderQueue;
use App\Models\SalesCreditLine;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineFinancial;
use App\Models\SalesOrderLineLayer;
use App\SalesCreditAllocation;
use Illuminate\Console\Command;

class DeleteDuplicateSalesOrderLines extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'patch:delete-duplicate-sales-order-lines';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Delete duplicate sales order line records';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        $uniqueFields = [
            'sales_order_id',
            'sales_channel_line_id',
            'product_listing_id',
            'product_id',
            'description',
            'quantity',
            'amount',
            'warehouse_id'
        ];

        $salesOrderLineUsageFieldsToSum = [
            BackorderQueue::class => ["backordered_quantity", "released_quantity"],
            SalesCreditLine::class => ["quantity"],
            SalesOrderFulfillmentLine::class => ["quantity"],
            SalesOrderLineFinancial::class => ["revenue", "revenue_allocated", "credits", "cogs", "cost_allocated", "cogs_returned"],
            SalesOrderLineLayer::class => ["quantity"],
        ];

        $select = "";
        foreach ($uniqueFields as $field){
            $select .= "COALESCE($field, '') as $field,";
        }
        $select = rtrim($select, ',');

        $salesOrderLineWithDuplication = SalesOrderLine::query()
            ->selectRaw($select)
            ->groupBy('hash')
            ->havingRaw('COUNT(*) > 1')
            ->get();

        $salesOrderLineWithDuplication->each(function ($salesOrderLine) use ($salesOrderLineUsageFieldsToSum, $uniqueFields) {
            $salesOrderLineUsagesQuery = SalesOrderLine::query();

            foreach ($uniqueFields as $field) {
                if (!empty($salesOrderLine->$field)) {
                    $salesOrderLineUsagesQuery->where($field, $salesOrderLine->$field);
                } else {
                    $salesOrderLineUsagesQuery->where(function ($query) use ($field) {
                        $query->whereNull($field)->orWhere($field, '')->orWhere($field, 0);
                    });
                }
            }

            $salesOrderLineUsagesQuery->orderBy('id');
            $totalQty = $salesOrderLineUsagesQuery->sum('quantity');


            $salesOrderLineUsages = $salesOrderLineUsagesQuery->get();

            if ($salesOrderLineUsages->count() === 0) {
                $this->output->warning("No sales order line usages found for sales order line with the following query: " . $salesOrderLineUsagesQuery->toSqlWithBindings());
                return false;
            }

            $salesOrderLineIdToKeep = $salesOrderLineUsages->first()->id;
            $salesOrderLineIdsToDelete = $salesOrderLineUsages->skip(1)->pluck('id');
            $this->info("Sales Order Line ID to keep: {$salesOrderLineIdToKeep}, Deleting " . $salesOrderLineIdsToDelete->count() . " sales order lines");
            $this->info("Total Qty: {$totalQty}");

            foreach ($salesOrderLineUsageFieldsToSum as $modelClass => $fieldsToSum) {
                foreach ($fieldsToSum as $fieldToSum) {
                    $sumValue = $modelClass::whereIn('sales_order_line_id', $salesOrderLineUsages->pluck('id'))->sum($fieldToSum);
                    $this->info("Sum Value: {$sumValue}");
                    $modelClass::where('sales_order_line_id', $salesOrderLineIdToKeep)->update([
                        $fieldToSum => $sumValue
                    ]);
                    $this->info("Updated {$fieldToSum} for {$modelClass}");
                }
                $modelClass::whereIn('sales_order_line_id', $salesOrderLineIdsToDelete)->delete();
                $this->info("Deleted {$modelClass}");
            }

            SalesOrderLine::whereIn('id', $salesOrderLineIdsToDelete)->delete();
        });
    }
}
