<?php

namespace App\Console\Commands;

use App\Actions\InventoryHealth\MoveMovementsFromSourceToTargetFifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\PurchaseOrderLine;
use App\Models\WarehouseTransferLine;
use App\Repositories\InventoryAdjustmentRepository;
use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Throwable;

class CorrectFifoForNegativeReceiptsCommand extends Command
{
    protected $signature = 'sku:correct-fifo-for-negative-receipts {adjustmentIds?}';
    protected $description = 'Correct FIFO Layers for Negative Receipts (Purchase Order or Warehouse Transfer)';

    private InventoryAdjustmentRepository $adjustments;
    private MoveMovementsFromSourceToTargetFifoLayer $moveMovementsFromSourceToTargetFifoLayer;

    public function __construct()
    {
        parent::__construct();
        $this->adjustments                              = app(InventoryAdjustmentRepository::class);
        $this->moveMovementsFromSourceToTargetFifoLayer = app(MoveMovementsFromSourceToTargetFifoLayer::class);
    }

    public function handle(): void
    {
        $negativeAdjustments = $this->getNegativeAdjustments();

        $negativeAdjustments->each(/**
         * @throws Throwable
         */ function (InventoryAdjustment $inventoryAdjustment) {
            $this->processInventoryAdjustment($inventoryAdjustment);
        });
    }

    private function getNegativeAdjustments(): Collection
    {
        $adjustmentIds = $this->argument('adjustmentIds')
            ? explode(',', $this->argument('adjustmentIds'))
            : null;

        $negativeAdjustments = $this->adjustments->getNegativeAdjustmentsForReceiptLines();

        if ($adjustmentIds) {
            $negativeAdjustments = $negativeAdjustments->whereIn('id', $adjustmentIds);
        }

        return $negativeAdjustments;
    }

    /**
     * @throws Throwable
     */
    private function processInventoryAdjustment(InventoryAdjustment $inventoryAdjustment): void
    {
        DB::transaction(function () use ($inventoryAdjustment) {
            /** @var PurchaseOrderLine|WarehouseTransferLine $line */
            $line = $inventoryAdjustment->link;
            $eligibleFifoLayers = $this->getEligibleFifoLayers($line);
            $usedFifoLayers = $inventoryAdjustment->inventoryMovements->pluck('layer');
            $incorrectFifoLayers = $usedFifoLayers->diff($eligibleFifoLayers);
            $incorrectInventoryMovements = $inventoryAdjustment->inventoryMovements->whereIn('layer_id',
                $incorrectFifoLayers->pluck('id'));

            if ($incorrectInventoryMovements->isEmpty()) {
                return;
            }

            $adjustmentDescription = "Inventory adjustment $inventoryAdjustment->id for $inventoryAdjustment->link_type ($inventoryAdjustment->link_id)";
            $this->warn("Processing " . $adjustmentDescription . ', which has incorrect FIFO layers...');

            foreach ($incorrectInventoryMovements as $incorrectInventoryMovement) {
                foreach ($eligibleFifoLayers as $eligibleFifoLayer) {
                    $quantityToMove = abs($incorrectInventoryMovement->quantity);
                    $this->info("Deducting " . abs($quantityToMove) . " from incorrect layer $incorrectInventoryMovement->layer_id to eligible layer $eligibleFifoLayer->id");
                    ($this->moveMovementsFromSourceToTargetFifoLayer)($quantityToMove, $eligibleFifoLayer, $incorrectInventoryMovement->layer);
                    $eligibleFifoLayer->refresh();
                    if ($eligibleFifoLayer->available_quantity > 0) {
                        $quantityToMoveToEligibleLayer        = min($quantityToMove,
                            $eligibleFifoLayer->available_quantity);
                        $incorrectInventoryMovement->layer_id = $eligibleFifoLayer->id;
                        $incorrectInventoryMovement->save();
                        $this->info("Moved $quantityToMoveToEligibleLayer to eligible layer $eligibleFifoLayer->id");
                        $incorrectInventoryMovement->layer->fulfilled_quantity -= $quantityToMoveToEligibleLayer;
                        $incorrectInventoryMovement->layer->save();
                        $eligibleFifoLayer->fulfilled_quantity += $quantityToMoveToEligibleLayer;
                        $eligibleFifoLayer->save();
                        $quantityToMove                        -= $quantityToMoveToEligibleLayer;
                    }

                    if ($quantityToMove <= 0) {
                        break;
                    }
                }
            }
            $this->warn("Done processing $adjustmentDescription");
        });
    }

    private function getEligibleFifoLayers(PurchaseOrderLine|WarehouseTransferLine $line): Collection
    {
        return match (true) {
            $line instanceof PurchaseOrderLine => $line->purchaseOrderShipmentReceiptLines->pluck('fifoLayers')->flatten(),
            $line instanceof WarehouseTransferLine => $line->shipmentLine->receiptLines->pluck('fifoLayers')->flatten(),
        };
    }
}