<?php

namespace App\Actions;

use App\Exceptions\BackorderNotFoundForReallocationDestinationException;
use App\Exceptions\BackorderReallocationQuantityOverageException;
use App\Exceptions\BackorderReallocationSourceInsufficientMovableQuantityException;
use App\Exceptions\InvalidDestinationForBackorderReallocationException;
use App\Exceptions\InventoryMovementTypeException;
use App\Exceptions\OversubscribedFifoLayerException;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\BackorderQueue;
use App\Models\BackorderQueueCoverage;
use App\Models\BackorderQueueRelease;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Repositories\BackorderQueueRepository;
use App\Services\StockTake\OpenStockTakeException;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use Throwable;

class ReallocateBackorderRelease
{
    private BackorderQueueRepository $backorders;

    protected array $sourceFifoLayersAndQuantities = [];
    protected array $releasesCreated = [];

    public function __construct(
        private readonly int $quantityToReallocate,
        public BackorderQueueRelease $sourceBackorderRelease,
        private readonly SalesOrder $destinationSalesOrder,
    )
    {
        $this->backorders = app(BackorderQueueRepository::class);
    }

    /**
     * Re-allocates the given quantity from the backorder queue release
     * to backorder queues on the given sales order.
     *
     * @throws Exception
     * @throws Throwable
     */
    public function handle(): array
    {
        $product = $this->sourceBackorderRelease->backorderQueue->salesOrderLine->product;
        $this->validateReallocation($this->getDestinationSalesOrderShortageQuantity());

        return DB::transaction(function () use ($product)
        {
            $this->takeQuantityFromSource();
            $this->applyQuantityToDestination();

            (new UpdateProductsInventoryAndAvgCost([$product->id]))->handle();

            return $this->releasesCreated;
        });
    }

    /**
     * @throws BackorderNotFoundForReallocationDestinationException
     */
    private function getDestinationSalesOrderShortageQuantity(): int
    {
        $productId = $this->sourceBackorderRelease->backorderQueue->salesOrderLine->product_id;

        $salesOrderLine = $this->destinationSalesOrder->salesOrderLines()
            ->where('product_id', $productId)
            ->whereHas('backorderQueue')
            ->first();

        if (!$salesOrderLine || !$salesOrderLine->backorderQueue) {
            throw new BackorderNotFoundForReallocationDestinationException("No backorder queue found for the destination sales order.");
        }

        return $salesOrderLine->backorderQueue->shortage_quantity;
    }

    /**
     * @throws BackorderReallocationSourceInsufficientMovableQuantityException
     * @throws BackorderReallocationQuantityOverageException
     */
    private function validateReallocationQuantity(int $quantityToReallocate, int $destinationShortageQuantity): void
    {
        if ($quantityToReallocate > $destinationShortageQuantity) {
            throw new BackorderReallocationQuantityOverageException(
                "You can't reallocate more quantity to the destination sales order than it needs. The destination sales order only needs $destinationShortageQuantity."
            );
        }

        $releasedQuantity = $this->sourceBackorderRelease->released_quantity;
        $fulfilledQuantity = $this->sourceBackorderRelease->backorderQueue->salesOrderLine->fulfilled_quantity;

        $availableQuantityToReallocateFromSource = $releasedQuantity - $fulfilledQuantity;

        if ($availableQuantityToReallocateFromSource < $this->quantityToReallocate) {
            throw new BackorderReallocationSourceInsufficientMovableQuantityException(
                "The backorder release being moved only has $availableQuantityToReallocateFromSource available to move out of requested $this->quantityToReallocate."
            );
        }
    }

    private function isSameOrder(SalesOrder $salesOrder, BackorderQueueRelease $release): bool
    {
        return $salesOrder->id === $release->backorderQueue->salesOrderLine->sales_order_id;
    }

    private function orderHasBackorders(SalesOrder $salesOrder, BackorderQueueRelease $sourceBackorderRelease): bool
    {
        $productId = $sourceBackorderRelease->backorderQueue->salesOrderLine->product_id;

        return $salesOrder->salesOrderLines()
            ->where('product_id', $productId)
            ->whereHas('backorderQueue', function (Builder $builder) {
                $builder->active();
            })
            ->exists();
    }

    /**
     * Reverses the given quantity of release from the
     * backorder queue released by the backorder queue release.
     *
     * @throws InventoryMovementTypeException
     * @throws OpenStockTakeException
     * @throws OversubscribedFifoLayerException
     */
    protected function takeQuantityFromSource(): void
    {
        $this->updateSourceBackorderQueue($this->sourceBackorderRelease->backorderQueue);
        $this->updateSourceBackorderRelease();

        if ($this->sourceBackorderRelease->isPurchaseReceipt()) {
            $this->adjustSourceCoverages();
        }

        $sourceSalesOrderLine = $this->sourceBackorderRelease->backorderQueue->salesOrderLine;
        $quantityTakenFromSalesOrderLine = min($this->quantityToReallocate, $sourceSalesOrderLine->quantity);

        $this->updateInventoryMovements($sourceSalesOrderLine, $quantityTakenFromSalesOrderLine, $this->sourceBackorderRelease->backorder_queue_id);

        $this->sourceFifoLayersAndQuantities = $this->reduceSalesOrderLineLayersQtyBy($sourceSalesOrderLine, $quantityTakenFromSalesOrderLine);
    }

    private function updateSourceBackorderQueue(BackorderQueue $sourceBackorderQueue): void
    {
        $sourceBackorderQueue->released_quantity = max(0, $sourceBackorderQueue->released_quantity - $this->quantityToReallocate);

        if (!$sourceBackorderQueue->priority) {
            $lastPriority = $this->backorders->getLastPriority($sourceBackorderQueue->salesOrderLine->product_id);
            $sourceBackorderQueue->setPriority($lastPriority + 1);
        }

        $sourceBackorderQueue->save();
    }

    private function updateSourceBackorderRelease(): void {
        $this->sourceBackorderRelease->released_quantity = max(0, $this->sourceBackorderRelease->released_quantity - $this->quantityToReallocate);

        if ($this->sourceBackorderRelease->released_quantity === 0) {
            $this->sourceBackorderRelease->delete();
        } else {
            $this->sourceBackorderRelease->save();
        }
    }

    /**
     * @throws InventoryMovementTypeException
     * @throws OpenStockTakeException
     */
    private function updateInventoryMovements(SalesOrderLine $salesOrderLine, int $quantityTaken, int $backorderQueueId): void
    {
        if ($quantityTaken === $salesOrderLine->quantity) {
            $salesOrderLine->inventoryMovements()->update([
                'layer_type' => BackorderQueue::class,
                'layer_id' => $backorderQueueId,
            ]);
            return;
        }

        $this->adjustActiveMovement($salesOrderLine, $quantityTaken, $backorderQueueId);
        $this->adjustReservationMovement($salesOrderLine, $quantityTaken, $backorderQueueId);
    }

    /**
     * @throws InventoryMovementTypeException
     * @throws OpenStockTakeException
     */
    private function adjustActiveMovement(SalesOrderLine $salesOrderLine, int $quantityTaken, int $backorderQueueId): void
    {
        $activeMovement = $salesOrderLine->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('quantity', '<', 0)
            ->first();

        if ($activeMovement) {
            $activeMovement->quantity = min(0, $activeMovement->quantity + $quantityTaken);

            if ($activeMovement->quantity === 0) {
                $activeMovement->delete();
            } else {
                $activeMovement->save();
            }

            $activeMovementCopy = $activeMovement->replicate();
            $activeMovementCopy->quantity = -$quantityTaken;
            $activeMovementCopy->backorder_queue = $backorderQueueId;
            $activeMovementCopy->save();
        }
    }

    /**
     * @throws OpenStockTakeException
     * @throws InventoryMovementTypeException
     */
    private function adjustReservationMovement(SalesOrderLine $salesOrderLine, int $quantityTaken, int $backorderQueueId): void
    {
        $reservationMovement = $salesOrderLine->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->where('quantity', '>', 0)
            ->first();

        if ($reservationMovement) {
            $reservationMovement->quantity = max(0, $reservationMovement->quantity - $quantityTaken);

            if ($reservationMovement->quantity === 0) {
                $reservationMovement->delete();
            } else {
                $reservationMovement->save();
            }

            $reservationMovementCopy = $reservationMovement->replicate();
            $reservationMovementCopy->quantity = $quantityTaken;
            $reservationMovementCopy->backorder_queue = $backorderQueueId;
            $reservationMovementCopy->save();
        }
    }

    /**
     * Reduces the line layer quantity by the given amount,
     * adjusting for fifo layer quantities as well.
     *
     * @param SalesOrderLine $sourceSalesOrderLine
     * @param int $quantityTaken
     * @return array
     * @throws OversubscribedFifoLayerException
     */
    public function reduceSalesOrderLineLayersQtyBy(SalesOrderLine $sourceSalesOrderLine, int $quantityTaken): array
    {
        $sourceLineLayers = $sourceSalesOrderLine->salesOrderLineLayers()->with('layer')->get();

        $fifoLayersAndQuantities = [];

        foreach ($sourceLineLayers as $sourceLineLayer) {
            $fifoLayer = $sourceLineLayer->layer;

            if (!($fifoLayer instanceof FifoLayer)) {
                continue;
            }

            if ($sourceLineLayer->quantity >= $quantityTaken) {
                $sourceLineLayer->quantity -= $quantityTaken;

                if ($sourceLineLayer->quantity === 0) {
                    $sourceLineLayer->delete();
                } else {
                    $sourceLineLayer->save();
                }

                $fifoLayer->fulfilled_quantity -= $quantityTaken;
                $fifoLayer->save();

                $fifoLayersAndQuantities[] = [
                    'layer' => $fifoLayer,
                    'quantity_released' => $quantityTaken,
                ];
                break;
            } else {
                $quantityToReduce = $sourceLineLayer->quantity;

                $fifoLayer->fulfilled_quantity -= $quantityToReduce;
                $fifoLayer->save();

                $fifoLayersAndQuantities[] = [
                    'layer' => $fifoLayer,
                    'quantity_released' => $quantityToReduce,
                ];

                $sourceLineLayer->delete();
                $quantityTaken -= $quantityToReduce;
            }
        }

        return $fifoLayersAndQuantities;
    }

    /**
     * Releases backorder queues on the given sales order for the effective quantity
     * taken from the given release.
     *
     * @throws Throwable
     */
    private function applyQuantityToDestination(): void
    {
        $sourceProductId = $this->sourceBackorderRelease->backorderQueue->salesOrderLine->product_id;
        $sourceWarehouseId = $this->sourceBackorderRelease->backorderQueue->salesOrderLine->warehouse_id;

        $destinationSalesOrderLines = $this->destinationSalesOrder->salesOrderLines()
            ->with('backorderQueue')
            ->where('product_id', $sourceProductId)
            ->where('warehouse_id', $sourceWarehouseId)
            ->whereHas('backorderQueue', function (Builder $builder) {
                $builder->active();
            })
            ->get()
            ->sortBy('backorderQueue.priority');

        $reallocatedQuantity = 0;

        foreach ($destinationSalesOrderLines as $destinationSalesOrderLine) {
            if (
                empty($this->sourceFifoLayersAndQuantities) ||
                $reallocatedQuantity >= $this->quantityToReallocate
            ) {
                break;
            }

            $reallocatedQuantity += $this->releaseBackorderQueue($destinationSalesOrderLine->backorderQueue);
        }
    }

    /**
     * @param BackorderQueue $destinationBackorder
     * @return int
     * @throws Throwable
     */
    private function releaseBackorderQueue(BackorderQueue $destinationBackorder): int
    {
        $totalReleased = 0;

        foreach ($this->sourceFifoLayersAndQuantities as $key => $layerInfo) {
            $fifoLayer = $layerInfo['layer'];
            $quantityToRelease = $layerInfo['quantity_released'];

            $release = $destinationBackorder->addRelease(
                $fifoLayer->link_id,
                $fifoLayer->link_type,
                $quantityToRelease,
                $fifoLayer,
                suppressCoveragesSync: true
            );

            if ($release->released_quantity === $quantityToRelease) {
                unset($this->sourceFifoLayersAndQuantities[$key]);
            } else {
                $this->sourceFifoLayersAndQuantities[$key]['quantity_released'] -= $release->released_quantity;
            }

            $totalReleased += $release->released_quantity;
            $this->releasesCreated[] = $release;

            if ($fifoLayer->link_type === PurchaseOrderShipmentReceiptLine::class) {
                $this->adjustDestinationCoverage($destinationBackorder, $fifoLayer->link, $release->released_quantity);
            }
        }

        return $totalReleased;
    }

    private function adjustDestinationCoverage(
        BackorderQueue $destinationBackorder,
        PurchaseOrderShipmentReceiptLine $link,
        int $releasedQuantity
    ): void {
        $purchaseOrderLineId = $link->purchaseOrderLine->id;

        $existingCoverage = $destinationBackorder->backorderQueueCoverages()
            ->where('purchase_order_line_id', $purchaseOrderLineId)
            ->first();

        if ($existingCoverage) {
            $existingCoverage->covered_quantity += $releasedQuantity;
            $existingCoverage->released_quantity += $releasedQuantity;
            $existingCoverage->save();
        } else {
            BackorderQueueCoverage::create([
                'backorder_queue_id' => $destinationBackorder->id,
                'purchase_order_line_id' => $purchaseOrderLineId,
                'covered_quantity' => $releasedQuantity,
                'released_quantity' => $releasedQuantity,
            ]);
        }
    }

    private function adjustSourceCoverages(): void
    {
        $receiptLine = $this->sourceBackorderRelease->link;

        if (!($receiptLine instanceof PurchaseOrderShipmentReceiptLine)) {
            return;
        }

        $purchaseOrderLine = $receiptLine->purchaseOrderShipmentLine->purchaseOrderLine;

        $coverages = $purchaseOrderLine->coveredBackorderQueues()
            ->where('backorder_queue_id', $this->sourceBackorderRelease->backorder_queue_id)
            ->get();

        $remainingQuantity = $this->quantityToReallocate;

        foreach ($coverages as $coverage) {
            if ($coverage->released_quantity > $remainingQuantity) {
                $coverage->covered_quantity -= $remainingQuantity;
                $coverage->released_quantity -= $remainingQuantity;
                $coverage->save();
                break;
            } else {
                $remainingQuantity -= $coverage->released_quantity;
                $coverage->delete();
            }

            if ($remainingQuantity <= 0) {
                break;
            }
        }
    }

    /**
     * @throws BackorderReallocationQuantityOverageException
     * @throws BackorderReallocationSourceInsufficientMovableQuantityException
     * @throws InvalidDestinationForBackorderReallocationException
     */
    private function validateReallocation(int $destinationShortageQuantity): void {
        $this->validateReallocationQuantity($this->quantityToReallocate, $destinationShortageQuantity);
        if (
            $this->isSameOrder($this->destinationSalesOrder, $this->sourceBackorderRelease) ||
            !$this->orderHasBackorders($this->destinationSalesOrder, $this->sourceBackorderRelease)
        ) {
            throw new InvalidDestinationForBackorderReallocationException("The destination sales order is not a valid destination for the reallocation.  It either is the same as the source or has no backorders.");
        }
    }
}
