<?php

namespace App\Actions\InventoryManagement;

use App\Data\FifoLayerAllocationData;
use App\Data\FifoLayerQuantityData;
use App\Exceptions\UnableToAllocateToFifoLayersException;
use App\Models\FifoLayer;
use App\Models\Product;
use App\Models\Warehouse;
use Exception;
use Illuminate\Support\Collection;

class GetFifoLayersForQuantity
{
    protected Product $product;
    protected Warehouse $warehouse;
    protected int $quantityToAllocate;
    protected bool $allowPartialAllocation;
    protected ?FifoLayer $excludeFifoLayer;
    protected Collection $activeFifoLayers;
    protected Collection $allocatedFifoLayers;

    /**
     * @throws UnableToAllocateToFifoLayersException
     */
    public function __invoke(
        Product $product,
        Warehouse $warehouse,
        int $quantityToAllocate,
        bool $allowPartialAllocation = false,
        ?FifoLayer $excludeFifoLayer = null,
    ): FifoLayerAllocationData
    {
        $this->product = $product;
        $this->warehouse = $warehouse;
        $this->quantityToAllocate = $quantityToAllocate;
        $this->allowPartialAllocation = $allowPartialAllocation;
        $this->excludeFifoLayer = $excludeFifoLayer;
        $this->activeFifoLayers = new Collection();
        $this->allocatedFifoLayers = new Collection();

        $this->getActiveFifoLayers();
        $this->allocateToFifoLayers();

        return FifoLayerAllocationData::from([
            'fifoLayerQuantities' => FifoLayerQuantityData::collection($this->allocatedFifoLayers),
            'remainingQuantityToAllocate' => $this->quantityToAllocate,
        ]);
    }

    /**
     * @throws UnableToAllocateToFifoLayersException
     */
    private function getActiveFifoLayers(): void
    {
        $query = $this->product->activeFifoLayers(useCache: false)
            ->where('warehouse_id', $this->warehouse->id);

        if ($this->excludeFifoLayer) {
            $query->where('id', '!=', $this->excludeFifoLayer->id);
        }

        $this->activeFifoLayers = $query->get();
        if ($this->activeFifoLayers->isEmpty())
        {
            if ($this->allowPartialAllocation) {
                return;
            }
            throw new UnableToAllocateToFifoLayersException("No active fifo layers found for product: {$this->product->sku} (ID: {$this->product->id}) and warehouse: {$this->warehouse->name} (ID: {$this->warehouse->id}).");
        }
    }

    /**
     * @throws UnableToAllocateToFifoLayersException
     * @throws Exception
     */
    private function allocateToFifoLayers(): void
    {
        while ($this->quantityToAllocate > 0)
        {
            /** @var FifoLayer $fifoLayer */
            $fifoLayer = $this->activeFifoLayers->shift();
            if (!$fifoLayer)
            {
                if ($this->allowPartialAllocation) {
                    return;
                }
                throw new UnableToAllocateToFifoLayersException('No more fifo layers to allocate to.  Still have ' . $this->quantityToAllocate . ' to allocate.');
            }
            $fifoLayerAllocationQuantity = min($fifoLayer->calculatedAvailableQuantity(), $this->quantityToAllocate);
            $this->allocatedFifoLayers->push(FifoLayerQuantityData::from([
                'fifoLayer' => $fifoLayer,
                'quantity' => $fifoLayerAllocationQuantity,
            ]));
            $this->quantityToAllocate -= $fifoLayerAllocationQuantity;
        }
    }


}
