<?php

namespace App\Actions\InventoryManagement;

use App\Actions\InventoryHealth\UpdateSalesOrderLineLayerCache;
use App\Exceptions\OversubscribedFifoLayerException;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\SalesOrderLine;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Throwable;

class DeleteSalesOrderReservation
{
    protected SalesOrderLine $salesOrderLine;
    protected Collection $inventoryMovementsDeleted;

    public function __construct(
        private readonly UpdateSalesOrderLineLayerCache $updateSalesOrderLineLayerCache,
    )
    {
    }

    /**
     * @throws Throwable
     */
    public function __invoke(
        SalesOrderLine $salesOrderLine
    ): Collection
    {
        $this->inventoryMovementsDeleted = collect();
        $this->salesOrderLine = $salesOrderLine;

        DB::transaction(function () {
            $this->deleteInventoryMovements();
            $this->updateFifoLayerCache();
        });

        return $this->inventoryMovementsDeleted;
    }

    /**
     * @throws Throwable
     */
    private function deleteInventoryMovements(): void
    {
        $this->salesOrderLine->inventoryMovements->each(function (InventoryMovement $inventoryMovement) {
            $this->inventoryMovementsDeleted->add($inventoryMovement->withoutRelations());
            $inventoryMovement->delete();
            if ($inventoryMovement->layer_type == FifoLayer::class) {
                ($this->updateSalesOrderLineLayerCache)($inventoryMovement->link, $inventoryMovement->layer, -$inventoryMovement->quantity);
            }
        });
    }

    private function updateFifoLayerCache(): void
    {
        $this->inventoryMovementsDeleted
            ->where('layer_type', FifoLayer::class)
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->groupBy(function(InventoryMovement $inventoryMovement) {
                return $inventoryMovement->layer_id;
            })
            ->each(/**
             * @throws OversubscribedFifoLayerException
             */ function (Collection $inventoryMovements, int $fifoLayerId) {
                $fifoLayer = FifoLayer::findOrFail($fifoLayerId);
                $quantity                      = abs($inventoryMovements->sum('quantity'));
                $fifoLayer->fulfilled_quantity -= $quantity;
                $fifoLayer->save();
            });
    }
}
