<?php

namespace App\Actions\InventoryManagement;

use App\Data\FifoLayerQuantityData;
use App\Exceptions\InventoryMovementTypeException;
use App\Exceptions\OversubscribedFifoLayerException;
use App\Exceptions\UnableToAllocateToFifoLayersException;
use App\Models\BackorderQueue;
use App\Models\InventoryMovement;
use App\Repositories\FifoLayerRepository;
use App\Repositories\InventoryHealthRepository;
use App\Services\StockTake\OpenStockTakeException;
use Illuminate\Support\Collection;
use Spatie\LaravelData\DataCollection;

class CreateActiveInventoryMovementFromReserved
{
    protected InventoryMovement $reservedMovement;
    protected DataCollection|FifoLayerQuantityData $fifoLayerData;
    protected Collection $createdMovements;

    public function __construct(
        private readonly GetFifoLayersForQuantity $getFifoLayersForQuantity,
        private readonly InventoryHealthRepository $health,
        private readonly FifoLayerRepository $fifoLayers,
    )
    {
    }

    /**
     * @throws OpenStockTakeException
     * @throws InventoryMovementTypeException
     * @throws OversubscribedFifoLayerException
     * @throws UnableToAllocateToFifoLayersException
     */
    public function __invoke(InventoryMovement $reservedMovement): Collection
    {
        $this->reservedMovement = $reservedMovement;
        $this->createdMovements = collect();

        $this->getFifoLayersNeeded();
        $this->createActiveInventoryMovements();
        $this->updateFifoLayerCache();

        return $this->createdMovements;
    }

    /**
     * @throws UnableToAllocateToFifoLayersException
     */
    private function getFifoLayersNeeded(): void
    {
        if (
            $this->reservedMovement->layer instanceof BackorderQueue ||
            $this->reservedMovement->layer->available_quantity >= $this->reservedMovement->quantity
        ) {
            $this->fifoLayerData = FifoLayerQuantityData::from([
                'fifoLayer' => $this->reservedMovement->layer, // TODO: this will cause data issue if type is backorderqueue
                'quantity' => $this->reservedMovement->quantity

            ]);
            return;
        }

        $this->fifoLayerData = ($this->getFifoLayersForQuantity)(
            $this->reservedMovement->product,
            $this->reservedMovement->warehouse,
            $this->reservedMovement->quantity
        )->fifoLayerQuantities;
    }

    /**
     * @throws OpenStockTakeException
     * @throws InventoryMovementTypeException
     */
    private function createActiveInventoryMovements(): void
    {
        if ($this->fifoLayerData instanceof FifoLayerQuantityData) {
            $this->createdMovements->push($this->health->createActiveInventoryMovementFromReservationMovement(
                $this->reservedMovement,
                $this->fifoLayerData->quantity,
                $this->fifoLayerData->fifoLayer,
            ));
            return;
        }
        $this->fifoLayerData->each(function (FifoLayerQuantityData $fifoLayerData) {
            $this->createdMovements->push($this->health->createActiveInventoryMovementFromReservationMovement(
                $this->reservedMovement,
                $fifoLayerData->quantity,
                $fifoLayerData->fifoLayer,
            ));
            $this->createdMovements->push($this->createNewReservationMovementFromAvailableFifoLayerData($fifoLayerData));
        });
        // Delete old reservations since new ones were created to correspond with active movements
        $this->reservedMovement->delete();
    }

    /**
     * @throws InventoryMovementTypeException
     * @throws OpenStockTakeException
     */
    private function createNewReservationMovementFromAvailableFifoLayerData(FifoLayerQuantityData $data): InventoryMovement
    {
        $newReservedMovement = $this->reservedMovement->replicate();
        $newReservedMovement->quantity = $data->quantity;
        $newReservedMovement->layer()->associate($data->fifoLayer);
        $newReservedMovement->save();

        return $newReservedMovement;
    }

    /**
     * @throws OversubscribedFifoLayerException
     */
    private function updateFifoLayerCache(): void
    {
        if ($this->fifoLayerData instanceof FifoLayerQuantityData) {
            $this->fifoLayers->validateFifoLayerCache($this->fifoLayerData->fifoLayer);
            return;
        }
        $this->fifoLayerData->each(function (FifoLayerQuantityData $fifoLayerData) {
            $this->fifoLayers->validateFifoLayerCache($fifoLayerData->fifoLayer);
        });
    }
}
