<?php

namespace App\Actions\InventoryHealth;

use App\Exceptions\InsufficientSalesOrderLineQuantityException;
use App\Exceptions\InventoryMovementTypeException;
use App\Exceptions\OversubscribedFifoLayerException;
use App\Exceptions\SupplierWarehouseCantHaveInventoryMovementsException;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\SalesOrderLine;
use App\Models\WarehouseTransferShipmentLine;
use App\Repositories\FifoLayerRepository;
use App\Repositories\InventoryMovementRepository;
use App\Services\InventoryManagement\Actions\MoveSalesOrderLineLayersQuantity;
use App\Services\StockTake\OpenStockTakeException;
use Throwable;

/*
 * Should prefer most recent usages to move.
 * Could also filter by type of usage if needed.
 *
 * Usages that are negative inventory adjustments that are linked are not movable.
 */

class MoveMovementsFromSourceToTargetFifoLayer
{
    protected int $quantityToMove;
    protected FifoLayer $sourceFifoLayer;
    protected FifoLayer $targetFifoLayer;

    public function __construct(
        private readonly InventoryMovementRepository $movements,
        private readonly FifoLayerRepository $layers,
    )
    {}

    /**
     * @throws OversubscribedFifoLayerException
     * @throws Throwable
     * @throws InsufficientSalesOrderLineQuantityException
     */
    public function __invoke(
        int $quantityToMove,
        FifoLayer $sourceFifoLayer,
        FifoLayer $targetFifoLayer,
    ): void
    {

        $this->quantityToMove = $quantityToMove;
        $this->sourceFifoLayer = $sourceFifoLayer;
        $this->targetFifoLayer = $targetFifoLayer;

        $this->move();
    }

    /**
     * @throws OversubscribedFifoLayerException
     * @throws Throwable
     * @throws InsufficientSalesOrderLineQuantityException
     */
    private function move(): void
    {
        $movableUsages = $this->layers->getRelocatableInventoryMovements($this->sourceFifoLayer);

        foreach ($movableUsages as $movement) {
            $this->moveMovement($movement);
            if ($this->quantityToMove <= 0) {
                break;
            }
        }
    }

    /**
     * @throws OversubscribedFifoLayerException
     * @throws Throwable
     * @throws InsufficientSalesOrderLineQuantityException
     */
    private function moveMovement(InventoryMovement $movement): void
    {
        $movableQuantity = min(abs($this->quantityToMove), abs($movement->quantity));

        $this->handleSalesOrderLineLinkType($movement);
        $this->handleWarehouseTransferShipmentLineLinkType($movement);

        $movement->layer_id = $this->targetFifoLayer->id;
        $movement->save();

        $this->updateFifoLayerFulfilledQuantityCaches($movableQuantity);

        $this->quantityToMove -= $movableQuantity;
    }

    /**
     * @throws Throwable
     * @throws InsufficientSalesOrderLineQuantityException
     */
    private function handleSalesOrderLineLinkType(InventoryMovement $movement): void
    {
        if ($movement->link_type !== SalesOrderLine::class) {
            return;
        }
        // Get corresponding reservation and fulfillment and move those too
        $movement = $this->movements->getCorrespondingReservationMovementForActiveSalesOrderLineMovement($movement);
        $movement->layer_id = $this->targetFifoLayer->id;
        $movement->save();

        $this->movements->getCorrespondingFulfillmentMovementsForReservationSalesOrderLineMovement($movement)
            ->each(function($movement) {
                $movement->layer_id = $this->targetFifoLayer->id;
                $movement->save();
            });
        (new MoveSalesOrderLineLayersQuantity($movement->link->load('salesOrderLineLayers'), abs($movement->quantity), $this->sourceFifoLayer, $this->targetFifoLayer))->handle();
        $this->triggerCogsUpdate($movement);
    }

    /**
     * @throws InventoryMovementTypeException
     * @throws OpenStockTakeException
     * @throws Throwable
     * @throws SupplierWarehouseCantHaveInventoryMovementsException
     */
    private function handleWarehouseTransferShipmentLineLinkType(InventoryMovement $movement): void {
        if ($movement->link_type !== WarehouseTransferShipmentLine::class) {
            return;
        }

        $addInTransitMovement = $this->movements->getCorrespondingAddInTransitMovementForActiveLineMovement($movement);
        $deductInTransitMovements = $this->movements->getCorrespondingDeductInTransitMovementForAddInTransitLineMovement($addInTransitMovement);

        $addInTransitMovement->layer_id = $this->targetFifoLayer->id;
        $addInTransitMovement->save();

        $deductInTransitMovements->each(function(InventoryMovement $deductInTransitMovement) {
            $deductInTransitMovement->layer_id = $this->targetFifoLayer->id;
            $deductInTransitMovement->save();
        });

        $movement->layer_id = $this->targetFifoLayer->id;
        $movement->save();
    }

    /**
     * @throws OversubscribedFifoLayerException
     */
    private function updateFifoLayerFulfilledQuantityCaches(int $quantityMoved): void
    {
        $this->targetFifoLayer->fulfilled_quantity += $quantityMoved;
        $this->targetFifoLayer->save(allowOverage: true);
        $this->sourceFifoLayer->fulfilled_quantity -= $quantityMoved;
        $this->sourceFifoLayer->save(allowOverage: true);
    }

    private function triggerCogsUpdate(InventoryMovement $movement): void
    {
        $movement->link->touch();
    }
}