<?php

namespace App\Actions\InventoryHealth;

use App\Actions\InventoryManagement\GetFifoLayersForQuantity;
use App\Actions\InventoryManagement\SplitInventoryMovement;
use App\Data\FifoLayerAllocationData;
use App\Data\FifoLayerQuantityData;
use App\Exceptions\CantRelocateInventoryMovementToSameWarehouseException;
use App\Exceptions\InsufficientSalesOrderLineQuantityException;
use App\Exceptions\InventoryMovementTypeException;
use App\Exceptions\OversubscribedFifoLayerException;
use App\Exceptions\SupplierWarehouseCantHaveInventoryMovementsException;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\SalesOrderLine;
use App\Models\WarehouseTransferShipmentLine;
use App\Repositories\FifoLayerRepository;
use App\Repositories\InventoryMovementRepository;
use App\Services\InventoryManagement\Actions\MoveSalesOrderLineLayersQuantity;
use App\Services\StockTake\OpenStockTakeException;
use DB;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Throwable;

class FifoLayerInventoryMovementRelocator
{
    protected int $quantityToRelocate;
    protected FifoLayer $sourceFifoLayer;
    protected FifoLayerAllocationData $fifoLayerAllocationData;

    public function __construct(
        private readonly InventoryMovementRepository $movements,
        private readonly FifoLayerRepository $layers,
        private readonly GetFifoLayersForQuantity $getFifoLayersForQuantity,
        private readonly SplitInventoryMovement $splitInventoryMovement,
    )
    {}

    /**
     * @throws OversubscribedFifoLayerException
     * @throws Throwable
     * @throws InsufficientSalesOrderLineQuantityException
     */
    public function __invoke(
        int $quantityToRelocate,
        FifoLayer $sourceFifoLayer,
    ): FifoLayerAllocationData
    {
        $this->quantityToRelocate = $quantityToRelocate;
        $this->sourceFifoLayer    = $sourceFifoLayer;
        $this->layers->validateFifoLayerCache($this->sourceFifoLayer);

        $this->fifoLayerAllocationData = ($this->getFifoLayersForQuantity)(
            $sourceFifoLayer->product,
            $sourceFifoLayer->warehouse,
            $quantityToRelocate,
            allowPartialAllocation: true,
            excludeFifoLayer: $sourceFifoLayer,
        );

        if ($this->fifoLayerAllocationData->fifoLayerQuantities->toCollection()->isEmpty()) {
            return $this->fifoLayerAllocationData;
        }

        DB::transaction(function () {
            $this->relocate();
            $this->updateCache();
        });

        return $this->fifoLayerAllocationData;
    }

    /**
     * @throws OversubscribedFifoLayerException
     * @throws Throwable
     * @throws InsufficientSalesOrderLineQuantityException
     */
    private function relocate(): void
    {
        $relocatableInventoryMovements = $this->layers->getRelocatableInventoryMovements($this->sourceFifoLayer);

        if ($relocatableInventoryMovements->isEmpty()) {
            customlog('inventory-fixes', "No relocatable inventory movements found for FIFO Layer ID {$this->sourceFifoLayer->id}", days: null);
            return;
        }

        foreach ($relocatableInventoryMovements as $relocatableInventoryMovement) {
            if ($this->quantityToRelocate <= 0) {
                return;
            }
            $this->relocateMovement($relocatableInventoryMovement);
        }
    }

    /**
     * @throws Throwable
     */
    private function relocateMovement(InventoryMovement $relocatableInventoryMovement): void
    {
        // If we don't need to relocate the full relocatable inventory movement, split it into a new movement with the quantity that
        // does need to relocate and use that new movement.

        $lastRelocationWasSplit = false;
        if ($this->quantityToRelocate < abs($relocatableInventoryMovement->quantity)) {
            customlog('inventory-fixes', "Movement ID $relocatableInventoryMovement->id needs to be split so that only $this->quantityToRelocate can be reallocated (This is the last relocation needed)", days: null);
            $quantityToSplit = abs($relocatableInventoryMovement->quantity) - $this->quantityToRelocate;
            // We are splitting the movement but keeping the one that gets relocated to be the original movement not the newly created one
            ($this->splitInventoryMovement)($relocatableInventoryMovement, $quantityToSplit);
            $relocatableInventoryMovement->refresh();
            $lastRelocationWasSplit = true;
        }

        $this->fifoLayerAllocationData->fifoLayerQuantities->each(function(FifoLayerQuantityData $fifoLayerQuantity) use (&$relocatableInventoryMovement, $lastRelocationWasSplit) {
            if ($fifoLayerQuantity->fifoLayer->id == $this->sourceFifoLayer->id) {
                throw new CantRelocateInventoryMovementToSameWarehouseException("Can't relocate inventory movement $relocatableInventoryMovement->id to the same FIFO Layer {$this->sourceFifoLayer->id}");
            }
            if ($fifoLayerQuantity->quantityReallocatable <= 0) {
                return true;
            }
            $originalRelocatableInventoryMovement = $relocatableInventoryMovement;
            $split = false;
            if (!$lastRelocationWasSplit) {
                customlog('inventory-fixes', "Relocating movement ID $relocatableInventoryMovement->id for $relocatableInventoryMovement->quantity from $relocatableInventoryMovement->layer_id to FIFO ID {$fifoLayerQuantity->fifoLayer->id}", days: null);
                $this->handleSalesOrderLineLinkType($fifoLayerQuantity->fifoLayer, $relocatableInventoryMovement);
                $this->handleWarehouseTransferShipmentLineLinkType($fifoLayerQuantity->fifoLayer, $relocatableInventoryMovement);
            }

            if ($fifoLayerQuantity->quantityReallocatable < abs($relocatableInventoryMovement->quantity)) {
                customlog('inventory-fixes', "Movement ID $relocatableInventoryMovement->id needs to be split so that only $fifoLayerQuantity->quantityReallocatable of " . abs($relocatableInventoryMovement->quantity) . " can be assigned to FIFO Layer ID {$fifoLayerQuantity->fifoLayer->id}", days: null);
                $quantityToSplit = $fifoLayerQuantity->quantityReallocatable;
                $relocatableInventoryMovement = ($this->splitInventoryMovement)($relocatableInventoryMovement, $quantityToSplit);
                $split = true;
            }


            if ($lastRelocationWasSplit) {
                customlog('inventory-fixes', "Relocating movement ID $relocatableInventoryMovement->id for $relocatableInventoryMovement->quantity from $relocatableInventoryMovement->layer_id to FIFO ID {$fifoLayerQuantity->fifoLayer->id}", days: null);
                $this->handleSalesOrderLineLinkType($fifoLayerQuantity->fifoLayer, $relocatableInventoryMovement);
                $this->handleWarehouseTransferShipmentLineLinkType($fifoLayerQuantity->fifoLayer, $relocatableInventoryMovement);
            }

            $relocatableInventoryMovement->layer_id = $fifoLayerQuantity->fifoLayer->id;
            $relocatableInventoryMovement->save();

            $this->updateFifoLayerFulfilledQuantityCaches($fifoLayerQuantity->fifoLayer, abs($relocatableInventoryMovement->quantity));

            $this->quantityToRelocate -= abs($relocatableInventoryMovement->quantity);
            $fifoLayerQuantity->quantityReallocatable -= abs($relocatableInventoryMovement->quantity);

            $relocatableInventoryMovement = $originalRelocatableInventoryMovement;

            // If the movement was entirely relocated, move on to next movement
            if (!$split) {
                return false;
            }
            return true;
        });
    }

    /**
     * @throws Throwable
     * @throws InsufficientSalesOrderLineQuantityException
     */
    private function handleSalesOrderLineLinkType(FifoLayer $targetFifoLayer, InventoryMovement $activeMovement): void
    {
        if ($activeMovement->link_type !== SalesOrderLine::class) {
            return;
        }
        // Get corresponding reservation and fulfillment and move those too
        try {
            $reservationMovement = $this->movements->getCorrespondingReservationMovementForActiveSalesOrderLineMovement($activeMovement);
        } catch (ModelNotFoundException) {
            // Reserved movement was not found, so no further movements to relocate.  The missing reservation movement will be
            // fixed in another fix
            customlog('inventory-fixes', "No corresponding reservation movement found for active sales order line movement ID $activeMovement->id", days: null);
            return;
        }

        // Move all corresponding fulfillment movements for reservation...
        $totalFulfillmentQuantityMoved = 0;
        $potentialFulfillmentMovements = $this->movements->getCorrespondingFulfillmentMovementsForReservationSalesOrderLineMovement($reservationMovement);
        while ($totalFulfillmentQuantityMoved < abs($reservationMovement->quantity) && $potentialFulfillmentMovements->isNotEmpty()) {
            $fulfillmentMovement = $potentialFulfillmentMovements->shift();
            $totalFulfillmentQuantityMoved += abs($fulfillmentMovement->quantity);
            $fulfillmentMovement->layer_id = $targetFifoLayer->id;
            $fulfillmentMovement->save();
        }

        $activeMovement->layer_id = $targetFifoLayer->id;
        $activeMovement->save();

        $reservationMovement->layer_id = $targetFifoLayer->id;
        $reservationMovement->save();

        try {
            (new MoveSalesOrderLineLayersQuantity($activeMovement->link->load('salesOrderLineLayers'), abs($activeMovement->quantity), $this->sourceFifoLayer, $targetFifoLayer))->handle();
        } catch (InsufficientSalesOrderLineQuantityException) {
        }
        $this->triggerCogsUpdate($activeMovement);
    }

    /**
     * @throws InventoryMovementTypeException
     * @throws OpenStockTakeException
     * @throws Throwable
     * @throws SupplierWarehouseCantHaveInventoryMovementsException
     */
    private function handleWarehouseTransferShipmentLineLinkType(
        FifoLayer $fifoLayer,
        InventoryMovement $movement
    ): void {
        if ($movement->link_type !== WarehouseTransferShipmentLine::class) {
            return;
        }

        $addInTransitMovement = $this->movements->getCorrespondingAddInTransitMovementForActiveLineMovement($movement);
        $deductInTransitMovements = $this->movements->getCorrespondingDeductInTransitMovementForAddInTransitLineMovement($addInTransitMovement);

        $addInTransitMovement->layer_id = $fifoLayer->id;
        $addInTransitMovement->save();

        $deductInTransitMovements->each(function(InventoryMovement $deductInTransitMovement) use ($fifoLayer) {
            $deductInTransitMovement->layer_id = $fifoLayer->id;
            $deductInTransitMovement->save();
        });

        $movement->layer_id = $fifoLayer->id;
        $movement->save();
    }

    /**
     * @throws OversubscribedFifoLayerException
     */
    private function updateFifoLayerFulfilledQuantityCaches(FifoLayer $targetFifoLayer, int $quantityMoved): void
    {
        $targetFifoLayer->fulfilled_quantity += $quantityMoved;
        $targetFifoLayer->save(); // don't allow overages here.  Don't want to create new problems
        $this->sourceFifoLayer->fulfilled_quantity -= $quantityMoved;
        $this->sourceFifoLayer->save(allowOverage: true); // in case it is still over-fulfilled
    }

    private function triggerCogsUpdate(InventoryMovement $movement): void
    {
        $movement->link->touch();
    }

    /**
     * @throws OversubscribedFifoLayerException
     */
    private function updateCache(): void
    {
        $this->layers->validateFifoLayerCache($this->sourceFifoLayer);
        $this->fifoLayerAllocationData->fifoLayerQuantities->each(function(FifoLayerQuantityData $fifoLayerQuantity) {
            $this->layers->validateFifoLayerCache($fifoLayerQuantity->fifoLayer);
        });
    }
}