<?php

namespace App\Actions\InventoryHealth;

use App\Actions\InventoryManagement\SplitInventoryMovement;
use App\Data\CreateBackorderQueueData;
use App\Models\BackorderQueue;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\SalesOrderLine;
use App\Repositories\InventoryMovementRepository;
use App\Services\InventoryManagement\BackorderManager;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Throwable;

class BackorderFromFifoLayer
{
    protected FifoLayer $fifoLayer;
    protected int $quantityToBackorder;
    protected int $quantityRemainingToBackorder;
    protected Collection $backorderableInventoryMovements;

    public function __construct(
        private readonly BackorderManager $backorderManager,
        private readonly InventoryMovementRepository $movements,
        private readonly SplitInventoryMovement $splitInventoryMovement,
    )
    {
    }

    public function __invoke(
        FifoLayer $fifoLayer,
        int $quantityToBackorder
    ): int
    {
        $this->fifoLayer = $fifoLayer;
        $this->quantityToBackorder = $quantityToBackorder;
        $this->quantityRemainingToBackorder = $quantityToBackorder;
        $this->backorderableInventoryMovements = new Collection();

        $this->getBackorderableInventoryMovements();
        $this->backorderMovements();

        // Return quantity backordered
        return $quantityToBackorder - $this->quantityRemainingToBackorder;
    }

    private function getBackorderableInventoryMovements(): void
    {
        $this->backorderableInventoryMovements = $this->movements->getActiveInventoryMovementsForUnfulfilledSalesOrderLinesForFifoLayer($this->fifoLayer);
    }

    private function backorderMovements(): void
    {
        $this->backorderableInventoryMovements->each(/**
         * @throws Throwable
         */ function (InventoryMovement $inventoryMovement) {
            if ($this->quantityRemainingToBackorder <= 0) {
                return;
            }
            if ($this->quantityRemainingToBackorder < abs($inventoryMovement->quantity)) {
                $quantityToSplit = $this->quantityRemainingToBackorder;
                $inventoryMovement = ($this->splitInventoryMovement)($inventoryMovement, $quantityToSplit);
            }
            $this->backorderInventoryMovement($inventoryMovement);
            $this->quantityRemainingToBackorder -= abs($inventoryMovement->quantity);
        });
    }

    /**
     * @throws Throwable
     */
    private function backorderInventoryMovement(InventoryMovement $activeInventoryMovement): void
    {
        DB::transaction(function () use ($activeInventoryMovement) {
            /** @var SalesOrderLine $salesOrderLine */
            $salesOrderLine = $activeInventoryMovement->link;
            $backorder = $this->backorderManager->newCreateBackorder(CreateBackorderQueueData::from([
                'salesOrderLine' => $salesOrderLine,
                'backordered_quantity' => abs($activeInventoryMovement->quantity),
            ]));

            $reservationMovement = $this->movements->getCorrespondingReservationMovementForActiveSalesOrderLineMovement($activeInventoryMovement);
            $activeInventoryMovement->layer_id = $backorder->id;
            $activeInventoryMovement->layer_type = BackorderQueue::class;
            $activeInventoryMovement->save();
            $reservationMovement->layer_id = $backorder->id;
            $reservationMovement->layer_type = BackorderQueue::class;
            $reservationMovement->save();
            $salesOrderLineLayer = $salesOrderLine->salesOrderLineLayers()
                ->where('layer_id', $this->fifoLayer->id)
                ->where('layer_type', FifoLayer::class)
                ->sole();
            if ($salesOrderLineLayer->quantity <= abs($activeInventoryMovement->quantity)) {
                $salesOrderLineLayer->delete();
            } else {
                $salesOrderLineLayer->quantity -= abs($activeInventoryMovement->quantity);
                $salesOrderLineLayer->save();
            }
            $this->fifoLayer->fulfilled_quantity -= abs($activeInventoryMovement->quantity);
            // Allowing because this is a reduction and can be part of fix of over-subscription
            $this->fifoLayer->save(allowOverage: true);
        });
    }
}
