<?php

namespace App\Abstractions\Integrations\SalesChannels;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Abstractions\Integrations\ClientResponseDataInterface;
use App\Abstractions\Integrations\IntegrationClientInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Data\SalesChannelProductImportMappingData;
use App\Data\SalesChannelProductToSkuProductMappingCollectionData;
use App\Data\SalesChannelProductToSkuProductMappingData;
use App\Data\SalesOrderLineMappingData;
use App\DTO\ProductListingDto;
use App\Exceptions\SalesChannelProductMappingException;
use App\Helpers;
use App\Jobs\MapSalesOrderLinesToSalesChannelProductsJob;
use App\Lib\SphinxSearch\SphinxSearch;
use App\Models\DataImportMapping;
use App\Models\TaskStatus\TaskStatus;
use App\Repositories\DataImportMappingRepository;
use App\Repositories\ProductListingRepository;
use App\Repositories\ProductRepository;
use App\Repositories\SalesOrderLineRepository;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Storage;
use Modules\WooCommerce\Entities\WooCommerceProduct;
use Spatie\LaravelData\DataCollection;
use Spatie\LaravelData\Optional;
use Throwable;

abstract class AbstractSalesChannelProductManager extends AbstractSalesChannelManager implements SalesChannelProductManagerInterface
{
    protected AbstractSalesChannelProduct $model;

    public Collection $mappings;

    protected ProductListingRepository $listings;

    protected SalesOrderLineRepository $salesOrderLines;

    public function __construct(
        protected IntegrationInstanceInterface $integrationInstance,
        protected IntegrationClientInterface $client,
        protected SalesChannelProductRepositoryInterface $productRepository
    ) {
        parent::__construct($integrationInstance, $client);
        $this->setDataMappings();
        $this->listings = app(ProductListingRepository::class);
        $this->salesOrderLines = app(SalesOrderLineRepository::class);
    }

    abstract protected function getProductRepository();

    abstract protected function postCreateSkuProducts();

    abstract protected function syncInventory(?array $ids = []): void;

    public function cacheProductListingQuantity(?array $productIds = []): void
    {
        if(! $this->integrationInstance instanceof AbstractSalesChannelIntegrationInstance ){
            return;
        }

        if($this->integrationInstance->hasInventoryLocations()){
            // Handle inventory locations.
            $this->listings->cacheListingQuantityForInventoryLocations(
                integrationInstance: $this->integrationInstance,
                productIds: $productIds
            );

            // Aggregate product inventory quantity from locations.
            $this->listings->cacheProductListingQuantityFromInventoryLocations($this->integrationInstance, $productIds);
        } else {
            // Handle product listings quantity.
            $this->listings->cacheProductListingQuantity($this->integrationInstance, $productIds);
        }

    }

    public function setDataMappings(): void
    {
        $this->mappings = app(DataImportMappingRepository::class)->getListingMappings($this->integrationInstance->id);
    }

    protected function setModel(AbstractSalesChannelProduct $model): void
    {
        $this->model = $model;
    }

    public function getStartDateForNew()
    {
        return $this->getProductRepository()->getStartDateForNew($this->integrationInstance);
    }

    /*
    |--------------------------------------------------------------------------
    | Fetch from API
    |--------------------------------------------------------------------------
    */

    protected function fetchProduct(string $productId): ?AbstractSalesChannelProduct
    {
        return $this->client->getProduct($productId);
    }

    protected function fetchProducts(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->client->getProducts($parameters);
    }

    /*
    |--------------------------------------------------------------------------
    | Save to DB
    |--------------------------------------------------------------------------
    */

    public function getProduct(string $productId): ?AbstractSalesChannelProduct
    {
        $productCollection = collect($this->fetchProduct($productId));

        if (! $productCollection->count()) {
            return null;
        }

        $this->productRepository->saveForIntegration($this->integrationInstance, $productCollection);

        return $this->productRepository->getProductFromId($this->integrationInstance, $productId);
    }

    public function refreshProducts(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        $responseDto = $this->fetchProducts($parameters);

        if (! $responseDto->collection->count()) {
            return $responseDto;
        }

        $this->productRepository->saveForIntegration($this->integrationInstance, $responseDto->collection->toCollection());

        return $responseDto;
    }

    /*
    |--------------------------------------------------------------------------
    | Transmute data to sku.io
    |--------------------------------------------------------------------------
    */

    /**
     * @throws Throwable
     */
    public function getSalesChannelProducts(array $productIds = [], bool $createForAll = false): Collection
    {
        throw_if(
            empty($productIds) && ! $createForAll,
            'Impossible parameters passed.  Must pass createForAll=true if no productIds are passed'
        );

        $builder = $this->model::query()
            ->doesntHave('productListing');

        // Only select the products that have ids explicitly passed.
        if (! empty($productIds)) {
            $builder->whereIn('id', $productIds);
        }

        $builder->orderBy('id');

        return $builder->get();
    }

    public function makeProductListingCollectionForImport(Collection $salesChannelProducts, bool $collectionHasProductIdColumn = false): DataCollection
    {

        return ProductListingDto::collection($salesChannelProducts->map(function (AbstractSalesChannelProduct $salesChannelProduct) use ($collectionHasProductIdColumn) {
            return ProductListingDto::from([
                'document_id' => $salesChannelProduct->id,
                'document_type' => get_class($salesChannelProduct),
                'listing_sku' => $salesChannelProduct[$salesChannelProduct::getSkuField()],
                // 'title' - not required, i should remove this field in the product listing refactor
                'sales_channel_id' => $this->integrationInstance->salesChannel->id,
                'sales_channel_listing_id' => $salesChannelProduct->applyDataMapping($salesChannelProduct, 'sales_channel_listing_id'),
                'product_id' => $collectionHasProductIdColumn ? $salesChannelProduct->product_id : null,
                'product' => $salesChannelProduct->getProductDto(),
            ]);
        }));
    }

    /**
     * @throws Throwable
     */
    public function createSkuProducts(array $productIds = [], bool $createForAll = false): void
    {
        $salesChannelProducts = $this->getSalesChannelProducts($productIds, $createForAll);

        $productListingCollection = ProductListingDto::collection($salesChannelProducts->map(function (AbstractSalesChannelProduct $salesChannelProduct) {
            // TODO: Handle is_fba field
            return ProductListingDto::from([
                'document_id' => $salesChannelProduct->id,
                'document_type' => get_class($salesChannelProduct),
                'listing_sku' => $salesChannelProduct[$salesChannelProduct::getSkuField()],
                'sales_channel_id' => $this->integrationInstance->salesChannel->id,
                'sales_channel_listing_id' => $salesChannelProduct->{$salesChannelProduct::getUniqueField()},
                'product' => $salesChannelProduct->getProductDto(), // Map update existing product if it exists
                'price' => $salesChannelProduct->applyDataMapping($salesChannelProduct, 'price'),
                'title' => $salesChannelProduct->applyDataMapping($salesChannelProduct, 'item_name'),
            ]);
        }));

        $productListingCollection = app(ProductListingRepository::class)->saveWithRelations($productListingCollection);

        if($productListingCollection->count() > 0)
        {
            MapSalesOrderLinesToSalesChannelProductsJob::dispatch(
                SalesOrderLineMappingData::collection(
                    $this->salesOrderLines->getUnmappedSalesOrderLinesForSalesChannel($this->integrationInstance, $productListingCollection)
                )
            );
        }

        $this->postCreateSkuProducts();
        SphinxSearch::rebuild('products');
    }

    /**
     * Hydrate mappings to:
     * - Find salesChannelProduct from listing_sku
     * - Find product (if exists) from mapped_sku
     * - Find productListing (if exists) from combination of salesChannelId and listing_sku
     */
    public function hydrateSalesChannelProductToSkuProductMappings(
        SalesChannelProductToSkuProductMappingCollectionData $productMappingData,
        AbstractSalesChannelProduct $salesChannelProduct,
        IntegrationInstanceInterface $integrationInstance
    ): Collection {
        $mappingData = $productMappingData->mapping->toCollection();

        $salesChannelProducts = $this->productRepository->getProductsFromUniqueSalesChannelProductIds(
            $mappingData->pluck('sales_channel_listing_id')->toArray(),
            $salesChannelProduct,
            $integrationInstance
        );

        $products = app(ProductRepository::class)->getProductsFromSkus(
            $mappingData->pluck('mapped_sku')->toArray(),
        );
        $productListings = $this->listings->getFromSalesChannelIdFromUniqueSalesChannelProductIds(
            $integrationInstance->salesChannel->id,
            $mappingData->pluck('sales_channel_listing_id')->toArray()
        );

        return $mappingData->map(function (SalesChannelProductToSkuProductMappingData $mapping) use ($salesChannelProducts, $salesChannelProduct, $products, $productListings) {
            if ($matchedSalesChannelProduct = $salesChannelProducts->where($salesChannelProduct::getUniqueField(), $mapping->sales_channel_listing_id)->first())
            {
                $mapping->salesChannelProduct = $matchedSalesChannelProduct;

            }
            $mapping->product = $products->where('sku', $mapping->mapped_sku)->first();
            $mapping->product_listing_id = $productListings->where('sales_channel_listing_id', $mapping->sales_channel_listing_id)->first()?->id;

            return $mapping;
        })->reject(fn ($mapping) => $mapping->salesChannelProduct instanceof Optional);
    }

    /**
     * @throws Throwable
     */
    public function mapSalesChannelProductsToSkuProducts(
        SalesChannelProductToSkuProductMappingCollectionData $productMappingData,
        AbstractSalesChannelProduct $salesChannelProduct,
        IntegrationInstanceInterface $integrationInstance
    ): void
    {

        $productMappingData = $this->hydrateSalesChannelProductToSkuProductMappings($productMappingData, $salesChannelProduct, $integrationInstance);

        $productMappingData = $this->unmapSalesChannelProducts($productMappingData);

        // Reject mappings that don't have a product
        $productMappingData = $productMappingData->reject(fn ($mapping) => !$mapping->product);

        $productListingCollection = ProductListingDto::collection($productMappingData->map(function (SalesChannelProductToSkuProductMappingData $mapping) {
            /** @var AbstractSalesChannelProduct $salesChannelProduct */
            $salesChannelProduct = $mapping->salesChannelProduct;
            if (! $mapping->product) {
                throw new SalesChannelProductMappingException('Trying to map '.$mapping->sales_channel_listing_id.' to a non existing product '.$mapping->mapped_sku);
            }

            return ProductListingDto::from([
                'id' => $mapping->product_listing_id, // Either new mapping (if null) or remapping
                'document_id' => $salesChannelProduct->id,
                'document_type' => get_class($salesChannelProduct),
                'listing_sku' => $salesChannelProduct->{$salesChannelProduct::getSkuField()},
                'sales_channel_id' => $this->integrationInstance->salesChannel->id,
                'sales_channel_listing_id' => $salesChannelProduct->{$salesChannelProduct::getUniqueField()},
                'product_id' => $mapping->product->id,
                'price' => $salesChannelProduct->applyDataMapping($salesChannelProduct, 'price') ?? 0,
                'title' => $salesChannelProduct->applyDataMapping($salesChannelProduct, 'item_name'),
            ]);
        }));

        $productListingCollection = $this->listings->saveWithRelations($productListingCollection);

        if($productListingCollection->count() > 0)
        {
            MapSalesOrderLinesToSalesChannelProductsJob::dispatch(
                SalesOrderLineMappingData::collection(
                    $this->salesOrderLines->getUnmappedSalesOrderLinesForSalesChannel($integrationInstance, $productListingCollection)
                )
            );
        }

        $this->postCreateSkuProducts();
    }

    private function unmapSalesChannelProducts(Collection $productMappingData): Collection
    {
        $unmappedProductListings = $productMappingData
            ->whereNull('mapped_sku')
            ->whereNotNull('product_listing_id')
            ->pluck('product_listing_id');
        if ($unmappedProductListings->count()) {
            $this->listings->deleteProductListingsForIds($unmappedProductListings->toArray());
            $productMappingData = $productMappingData->reject(fn ($mapping) => $unmappedProductListings->contains($mapping->product_listing_id));
        }

        return $productMappingData;
    }

    /**
     * @throws Throwable
     */
    public function importMappings(SalesChannelProductImportMappingData $data): void
    {
        $mappings = Helpers::csvFileToCollection(Storage::disk('sales-channel-product-mappings')->path($data->stored_name));

        /** @var AbstractSalesChannelIntegrationInstance $integrationInstance */
        $integrationInstance = $this->integrationInstance;

        $uniqueProductField = app($integrationInstance::getProductClass())::getUniqueField();

        $mappings = $mappings->map(function ($row) use ($uniqueProductField) {
            $row['sales_channel_listing_id'] = $row[$uniqueProductField];
            return $row;
        });

        $productMappingData = SalesChannelProductToSkuProductMappingCollectionData::from([
            'mapping' => SalesChannelProductToSkuProductMappingData::collection($mappings),
        ]);

        $this->mapSalesChannelProductsToSkuProducts($productMappingData, $this->model, $integrationInstance);
    }
}
