<?php

namespace App\Abstractions\Integrations\SalesChannels;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Data\CreateSkuProductsFromSalesChannelData;
use App\Data\SalesChannelProductImportMappingData;
use App\Data\SalesChannelProductToSkuProductMappingCollectionData;
use App\DataTable\DataTable;
use App\Http\Controllers\Controller;
use App\Jobs\ImportSalesChannelProductMappingsJob;
use App\Models\Product;
use App\Models\SalesChannelProductTemplate;
use App\Repositories\DataImportMappingRepository;
use App\Response;
use Exception;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Http\Request;
use Modules\Ebay\Entities\EbayLegacyProduct;
use Modules\Ebay\Entities\EbayProduct;
use Spatie\LaravelData\Optional;
use Throwable;

abstract class AbstractSalesChannelProductController extends Controller
{
    use DataTable;

    abstract protected function getModel(): string;

    abstract protected function getResource(): string;

    public function abstractShow(SalesChannelProductInterface $product)
    {
        return $this->response->addData($this->getResource()::make($product));
    }

    /**
     * @throws Exception|Throwable
     */
    public function abstractCreateSkuProducts(CreateSkuProductsFromSalesChannelData $data, IntegrationInstanceInterface $integrationInstance): Response
    {
        if (!$data->create_all_products && empty($data->ids)) {
            // Use filters to generate ids
            $data->ids = $this->getIdsFromFilters($this->getModel(), request(), $integrationInstance);
        }

        //Persist mappings only if exists in request payload
        if (! ($data->mappings instanceof Optional)) {
            app(DataImportMappingRepository::class)->persistMapping($integrationInstance, 'listings', $data->mappings->toArray());
        }

        if (! empty($data->filters)) {
            $request = new Request($data->toArray());
            $data->ids = $this->getIdsFromFilters($this->getModel(), $request, $integrationInstance);
        }

        //Create an instance for product manager
        $manager = app($this->getModel())->manager($integrationInstance);

        //Create SKU products
        $manager->createSkuProducts(($data->ids ?: []), $data->create_all_products);

        return $this->response->setMessage('Added to queue, products will be created shortly.');
    }

    /**
     * @throws Throwable
     */
    public function abstractMapProducts(SalesChannelProductToSkuProductMappingCollectionData $data, IntegrationInstanceInterface $integrationInstance): Response
    {
        /** @var SalesChannelProductInterface $salesChannelProduct */
        $salesChannelProduct = app($this->getModel());

        $salesChannelProduct->manager($integrationInstance)->mapSalesChannelProductsToSkuProducts($data, app($this->getModel()), $integrationInstance);

        $this->response->addData('Successfully mapped sales channel products to sku products');

        return $this->response;
    }

    public function abstractRefresh(ShouldQueue $refreshJob): Response
    {
        dispatch($refreshJob)->onQueue('sales-channels');

        return $this->response->setMessage('Added to queue, products will be refreshed shortly.');
    }

    public function abstractImportMappings(SalesChannelProductImportMappingData $data, IntegrationInstanceInterface $integrationInstance): Response
    {
        dispatch(
            new ImportSalesChannelProductMappingsJob(
                $data,
                app($this->getModel())->manager($integrationInstance)
            )
        )->onQueue('sales-channels');

        return $this->response->setMessage('Added to queue, mappings will be imported shortly.');
    }

    /**
     * @throws Exception
     */
    public function abstractSyncInventory(Request $request, IntegrationInstanceInterface $integrationInstance): Response
    {
        $request->validate([
            'ids' => 'sometimes|array',
        ]);
        app(EbayLegacyProduct::class)->manager($integrationInstance)->syncInventory($request->ids);

        return $this->response->setMessage('Successfully submitted to sales channel.');
    }

    public function abstractHydrateProductTemplate(Request $request, IntegrationInstanceInterface $integrationInstance): Response
    {
        $request->validate([
            'product_id' => 'required|integer|exists:products,id',
            'sales_channel_product_template_id' => 'required|integer|exists:sales_channel_product_templates,id',
        ]);

        $product = Product::find($request->product_id);
        $template = SalesChannelProductTemplate::find($request->sales_channel_product_template_id);

        $data = app(EbayLegacyProduct::class)->manager($integrationInstance)->hydrateProductTemplate($product, $template);

        return $this->response->addData($data);
    }

    public function abstractBuildPayload(Request $request, IntegrationInstanceInterface $integrationInstance): Response
    {
        $request->validate([
            'fields' => 'required|array',
        ]);

        $data = app(EbayLegacyProduct::class)->manager($integrationInstance)->buildPayload($request->input('fields'));

        return $this->response->addData($data);
    }

    public function abstractPublish(Request $request, IntegrationInstanceInterface $integrationInstance): Response
    {
        $request->validate([
            'product_id' => 'required|integer|exists:products,id',
            'payload' => 'required|string',
            'metadata' => 'sometimes|array',
        ]);

        $product = Product::find($request->product_id);

        $publishProductListingResponseData = app(EbayLegacyProduct::class)->manager($integrationInstance)->publish($product, $request->payload, $request->metadata ?? []);

        $data = $publishProductListingResponseData->productListing->toArray();
        $data['response'] = $publishProductListingResponseData->response;

        return $this->response->addData($data);
    }
}
