<?php

namespace App\Abstractions\Integrations\SalesChannels;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\IntegrationModelInterface;
use App\DataTable\DataTableModelInterface;
use App\DataTable\DataTableModelTrait;
use App\DataTable\Exports\DataTableExporter;
use App\DTO\ProductPricingDto;
use App\Exporters\MapsExportableFields;
use App\Http\Resources\ProductListingResource;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\ProductListing;
use App\Repositories\ProductPricingTierRepository;
use Awobaz\Compoships\Compoships;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Kirschbaum\PowerJoins\PowerJoins;
use Spatie\LaravelData\DataCollection;

/**
 * @property-read IntegrationInstanceInterface $integrationInstance
 * @property-read ProductListing $productListing
 */
abstract class AbstractSalesChannelProduct extends Model
    implements
    ActiveSalesChannelProductInterface,
    DataTableModelInterface,
    IntegrationModelInterface,
    MapsExportableFields,
    SalesChannelProductInterface
{
    use BulkImport;
    use Compoships;
    use DataTableModelTrait;
    use HasFilters;
    use HasSort;
    use PowerJoins;

    protected $guarded = [];

    protected $casts = [
        'json_object' => 'array',
    ];

    const BULK_THRESHOLD = 200;

    abstract protected static function newFactory();

    abstract public static function getSkuField(): string;

    abstract public static function getPriceField(): string;

    abstract public static function getIntegrationName(): string;

    abstract public static function getLastModified(): ?string;

    abstract public function getListingUrl(): ?string;

    abstract public function isUsed(): bool|array;

    public static function getProductListingResource(): string
    {
        return ProductListingResource::class;
    }

    public function integrationInstance(): BelongsTo
    {
        return $this->belongsTo(IntegrationInstanceInterface::class);
    }

    public function productListing(): MorphOne
    {
        return $this->morphOne(ProductListing::class, 'document');
    }

    public function generalFilterableColumns(): array
    {
        return ['id', $this::getSkuField()];
    }

    public static function getExportableFields(): array
    {
        $exportableFields = [];
        foreach (static::getAllColumnsAndTypes() as $column => $type) {
            $exportableFields[$column] = DataTableExporter::makeExportableField($column, false);
        }

        return array_merge($exportableFields, [
                'product.id' => DataTableExporter::makeExportableField('product.id', false),
                'product.name' => DataTableExporter::makeExportableField('product.name', false),
                'product.sku' => DataTableExporter::makeExportableField('product.sku', false),
                'mapped_sku' => DataTableExporter::makeExportableField('mapped_sku', false),
                'mapped_at' => DataTableExporter::makeExportableField('mapped_at', false),
        ]);
    }

    public function getPricing(): DataCollection
    {
        return ProductPricingDto::collection([ProductPricingDto::from([
            'price' => $this->applyDataMapping($this, 'price') ?? 0,
            'product_pricing_tier_id' => app(ProductPricingTierRepository::class)->getDefault(true)->id,
        ])]);
    }

    public function scopeFilterProductListing(Builder $builder, array $relation, string $operator, $value, $conjunction): Builder
    {
        // without whereHas product
        if ($operator == 'isEmpty') {
            return $builder->whereDoesntHave('productListing');
        } elseif ($operator == 'isNotEmpty') {
            return $builder->whereHas('productListing');
        } else {
            return $builder->whereHas('productListing.product', function ($query) use ($operator, $value) {
                $query->filterKey('sku', $operator, $value);
            });
        }
    }
}
