<?php

namespace App\Abstractions\Integrations\SalesChannels;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Abstractions\Integrations\ClientResponseDataInterface;
use App\Abstractions\Integrations\IntegrationClientInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Data\UpdateSalesOrderData;
use App\Data\UpdateSalesOrderPayloadData;
use App\DTO\SalesOrderDto;
use App\Jobs\AutomatedSalesOrderFulfillmentJob;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Repositories\SalesOrder\SalesOrderRepository;
use App\Repositories\SalesOrderFulfillmentRepository;
use App\Services\SalesOrder\Fulfillments\AutomatedWarehouseFulfillment;
use App\Services\SalesOrder\SalesOrderManager;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\WooCommerce\Entities\WooCommerceOrder;
use Spatie\LaravelData\DataCollection;
use Throwable;

abstract class AbstractSalesChannelOrderManager extends AbstractSalesChannelManager implements SalesChannelOrderManagerInterface
{
    protected AbstractSalesChannelOrder $model;

    protected SalesOrderRepository $salesOrderRepository;

    protected SalesOrderFulfillmentRepository $salesOrderFulfillmentRepository;

    public function __construct(
        protected IntegrationInstanceInterface $integrationInstance,
        protected IntegrationClientInterface $client,
        protected SalesChannelOrderRepositoryInterface $orderRepository,
    ) {
        $this->salesOrderRepository = app(SalesOrderRepository::class);
        $this->salesOrderFulfillmentRepository = app(SalesOrderFulfillmentRepository::class);

        parent::__construct($integrationInstance, $client);
    }

    abstract protected function getOrderRepository();

    protected function setModel(AbstractSalesChannelOrder $model): void
    {
        $this->model = $model;
    }

    protected function getStartDateForNew()
    {
        return $this->getOrderRepository()->getStartDateForNew($this->integrationInstance);
    }

    /*
    |--------------------------------------------------------------------------
    | Fetch from API
    |--------------------------------------------------------------------------
    */

    protected function fetchOrder(string $orderId): ClientResponseDataInterface
    {
        return $this->client->getOrder($orderId);
    }

    protected function fetchOrders(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->client->getOrders($parameters);
    }

    /*
    |--------------------------------------------------------------------------
    | Upload to API
    |--------------------------------------------------------------------------
    */

    protected function fulfillOrder(ApiDataTransformerInterface $parameters)
    {
        return $this->client->fulfillOrder($parameters);
    }

    public function fulfillOrders(ApiDataTransformerInterface $parameters)
    {
        return $this->client->fulfillOrders($parameters);
    }

    /*
    |--------------------------------------------------------------------------
    | Save to DB
    |--------------------------------------------------------------------------
    */

    public function getOrder(string $orderId): ?AbstractSalesChannelOrder
    {
        $orderCollection = $this->fetchOrder($orderId)->collection;

        if (! $orderCollection->count()) {
            return null;
        }

        $this->orderRepository->save($this->integrationInstance, $orderCollection->toCollection());

        $this->postProcess($orderId);

        return $this->orderRepository->getOrderFromId($this->integrationInstance, $orderId);
    }

    /**
     * @throws Throwable
     */
    public function refreshOrders(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        $responseDto = $this->fetchOrders($parameters);

        if (! $responseDto->collection->count()) {
            #return $responseDto;
        }
        $responseDto->collection = $this->preOrdersProcess($responseDto->collection);

        $this->orderRepository->save($this->integrationInstance, $responseDto->collection->toCollection());

        $this->postProcess();

        $this->handleCancelSalesOrders($responseDto->collection->toCollection());

        if ($this->integrationInstance->supportsUpdates()) {
            $this->updateSkuOrders(updateForAll: true);
        }

        return $responseDto;
    }

    public function fulfillOrderForSalesOrder(SalesOrder $salesOrder, ApiDataTransformerInterface $parameters): void
    {
        if ($this->fulfillOrder($parameters)) {
            $this->salesOrderRepository->markAllSalesOrderFulfillmentsAsSubmittedToSalesChannel($salesOrder);
        }
    }

    public function fulfillOrderForSalesOrderFulfillment(SalesOrderFulfillment $salesOrderFulfillment, ?ApiDataTransformerInterface $parameters): void
    {
        if ($this->fulfillOrder($parameters)) {
            $this->salesOrderFulfillmentRepository->markAsSubmittedToSalesChannel($salesOrderFulfillment);
        }
    }

    abstract protected function postProcess(?string $orderId = null);

    abstract protected function postProcessItems();

    /*
    |--------------------------------------------------------------------------
    | Transmute data to sku.io
    |--------------------------------------------------------------------------
    */

    /**
     * @throws Throwable
     */
    public function getSalesChannelOrders(array $orderIds = [], bool $createForAll = false, $enforceDateRestriction = true): EloquentCollection
    {
        $builder = $this->getSalesChannelOrdersQuery($orderIds, $createForAll, $enforceDateRestriction);

        return $builder->get();
    }

    /**
     * @throws Throwable
     */
    public function getSalesChannelOrdersForUpdate(array $orderIds = [], bool $createForAll = false): EloquentCollection
    {
        $builder = $this->getSalesChannelOrdersForUpdateQuery($orderIds, $createForAll);

        return $builder->get();
    }

    /**
     * @throws Throwable
     */
    public function getSalesChannelOrdersQuery(array $orderIds = [], bool $createForAll = false, $enforceDateRestriction = true): Builder
    {
        throw_if(
            empty($orderIds) && ! $createForAll,
            'Impossible parameters passed.  Must pass createForAll=true if no orderIds are passed'
        );

        $builder = $this->model::with([
            'orderItems' => function ($query) {
                $query->withProduct($this->integrationInstance);
            },
            'orderItems.product',
            'integrationInstance.salesChannel',
        ])
            ->currencyExists()
            ->has('orderItems')
            ->where('integration_instance_id', $this->integrationInstance->id)
            ->whereDoesntHave('salesOrder');

        if ($enforceDateRestriction) {
            $builder->where(function (Builder $builder) {
                $builder->where(function (Builder $builder) {
                    $builder->where($this->model::getOrderDate(), '>=', $this->integrationInstance->integration_settings['start_date']);
                    $builder->orWhereNull($this->model::getOrderDate());
                });
                $builder->dateRestrictionException();
            });
        }

        // Only select the orders that have ids explicitly passed.  If ids are explicitly passed, then we include archived orders
        if (! empty($orderIds)) {
            $builder->whereIn('id', $orderIds);
        } else {
            $builder->whereNull('archived_at');
        }

        $builder->orderBy('id');

        return $builder;
    }

    /**
     * @throws Throwable
     */
    public function getSalesChannelOrdersForUpdateQuery(array $orderIds = [], bool $createForAll = false): Builder
    {
        throw_if(
            empty($orderIds) && ! $createForAll,
            'Impossible parameters passed.  Must pass createForAll=true if no orderIds are passed'
        );

        \DB::enableQueryLog();
        $builder = $this->model::with([
            'orderItems' => function ($query) {
                $query->withProduct($this->integrationInstance);
            },
            'orderItems.product',
            'integrationInstance.salesChannel',
        ])
            ->currencyExists()
            ->has('orderItems')
            ->where('integration_instance_id', $this->integrationInstance->id)
            ->whereHas('salesOrder', function (Builder $query) {
                $query->whereColumn('last_synced_from_sales_channel_at', '<', $this->model->getTable() . '.updated_at');
                $query->orWhereNull('last_synced_from_sales_channel_at');
            });

        // Only select the orders that have ids explicitly passed.  If ids are explicitly passed, then we include archived orders
        if (! empty($orderIds)) {
            $builder->whereIn('id', $orderIds);
        } else {
            $builder->whereNull('archived_at');
        }

        $builder->orderBy('id');

        return $builder;
    }

    /**
     * @throws Throwable
     */
    public function createSkuOrders(array $orderIds = [], bool $createForAll = false, bool $enforceDateRestriction = true): void
    {
        $salesChannelOrders = $this->getSalesChannelOrders($orderIds, $createForAll, $enforceDateRestriction);

        if ($salesChannelOrders->isEmpty()) {
            return;
        }

        $salesOrderCollection = SalesOrderDto::collection($salesChannelOrders->map(function (AbstractSalesChannelOrder $salesChannelOrder) {
            $salesOrderDto = $salesChannelOrder->getSalesOrderDto();
            $salesOrderDto->sales_channel_id = $this->integrationInstance->salesChannel->id;
            $salesOrderDto->store_id = $this->integrationInstance->salesChannel->store_id;
            return $salesOrderDto;
        }));

        $salesOrderCollection = app(SalesOrderRepository::class)->saveWithRelations($salesOrderCollection);

        // Dispatch automated fulfillment job for each sales order
        foreach ($salesOrderCollection as $salesOrderData) {
            $salesOrder = SalesOrder::find($salesOrderData['id']);
            dispatch(new AutomatedSalesOrderFulfillmentJob($salesOrder));
        }
    }

    /**
     * @throws Throwable
     *
     * For now, we just update order and payment status... needs further consideration for update logic
     */
    public function updateSkuOrders(array $orderIds = [], bool $updateForAll = false): void
    {
        $salesChannelOrders = $this->getSalesChannelOrdersForUpdate($orderIds, $updateForAll);

        if ($salesChannelOrders->isEmpty()) {
            return;
        }

        $salesOrderManager = app(SalesOrderManager::class);
        $salesChannelOrders->each(function (AbstractSalesChannelOrder $salesChannelOrder) use ($salesOrderManager) {
            $orderStatuses = $salesChannelOrder->getSalesOrderStatusesDto();
            $salesOrderManager->updateOrder(UpdateSalesOrderData::from([
                'salesOrder' => $salesChannelOrder->salesOrder,
                'payload' => UpdateSalesOrderPayloadData::from([
                    'order_status' => $orderStatuses->order_status,
                    'payment_status' => $orderStatuses->payment_status,
                    'last_synced_from_sales_channel_at' => $salesChannelOrder->updated_at,
                ]),
            ]));
        });
    }

    public function preOrdersProcess(DataCollection $orders): DataCollection
    {
        return $orders;
    }

    public function handleCancelSalesOrders(Collection $collection): void
    {
        $this->orderRepository->getCanceledOrders(
            $this->integrationInstance,
            $collection
        )
        ->each(function (AbstractSalesChannelOrder $salesChannelOrder) {
            $salesChannelOrder->salesOrder->cancel();
        });
    }
}
