<?php

namespace App\Abstractions\Integrations\SalesChannels;

use App\Abstractions\Integrations\Data\StoreSalesChannelIntegrationDataInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Data\IntegrationInstanceUpdateResponseData;
use App\DataTable\DataTable;
use App\Http\Controllers\Controller;
use App\Jobs\CacheProductListingPriceJob;
use App\Jobs\GenerateCacheProductListingQuantityJob;
use App\Repositories\SalesChannelRepository;
use App\Response;
use App\Services\Accounting\AccountingTransactionManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Data\AmazonIntegrationSettingsData;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Throwable;

abstract class AbstractSalesChannelIntegrationInstanceController extends Controller
{
    use DataTable;

    protected SalesChannelRepository $salesChannelRepository;

    public function __construct()
    {
        $this->salesChannelRepository = app(SalesChannelRepository::class);
        parent::__construct();
    }

    abstract protected function getModel(): string;

    abstract protected function getResource(): string;

    public function abstractShow(IntegrationInstanceInterface $integrationInstance): Response
    {
        $integrationInstance->load([
            'salesChannel',
            'integration',
        ]);

        return $this->response->addData($this->getResource()::make($integrationInstance));
    }

    /**
     * @throws Throwable
     */
    public function abstractStore(StoreSalesChannelIntegrationDataInterface $data): Response
    {
        return DB::transaction(function () use ($data) {
            $integrationInstance = $this->integrationInstanceRepository->save($data);
            $this->salesChannelRepository->saveFromIntegrationInstance($integrationInstance);

            return $this->response->addData($integrationInstance);
        });
    }

    /**
     * @throws Throwable
     */
    public function abstractUpdate(StoreSalesChannelIntegrationDataInterface $data, IntegrationInstanceInterface $integrationInstance): IntegrationInstanceUpdateResponseData
    {
        $existingIntegrationSettings = $integrationInstance->integration_settings;
        $newIntegrationSettings = $data->integration_settings->toArray();

        // integrationSettingsChanged is set to the changed portion of the payload.
        $integrationSettingsChanged = arrayRecursiveDiff(
            $newIntegrationSettings,
            $existingIntegrationSettings,
        );

        /*
         * This is temporary solution to handle the problem of new data coming from the controller overriding all integration settings.  Each integration
         * has different fields needed too so we have to call out the specific data class I believe.  In the future, we should handle things so that
         * there is not an override of integration settings in general so that this conditional will not be needed.
         */
        if ($integrationInstance instanceof AmazonIntegrationInstance) {
            $data->integration_settings = AmazonIntegrationSettingsData::from(array_merge($existingIntegrationSettings, $integrationSettingsChanged));
        }

        DB::transaction(function () use ($integrationInstance, $data) {
            $integrationInstance->update($data->toArray());
            app(AccountingTransactionManager::class)->syncAccountingTransactionsSalesOrderIsSyncEnabledStatus($integrationInstance);
        });

        if (isset($integrationSettingsChanged['start_date'])) {
            $integrationInstance->setDate('start_date');
        }

        if (isset($integrationSettingsChanged['store_id'])) {
            $integrationInstance
                ->salesChannel->update(['store_id' => $integrationSettingsChanged['store_id']]);
        }

        if (isset($integrationSettingsChanged['pricing']['masterOfPrice']['name'])) {
            dispatch(new CacheProductListingPriceJob($integrationInstance));
        }
        if (isset($integrationSettingsChanged['inventory'])) {
            dispatch(new GenerateCacheProductListingQuantityJob($integrationInstance));
        }

        $this->response->addData($this->getResource()::make($integrationInstance));

        return IntegrationInstanceUpdateResponseData::from([
            'response' => $this->response,
            'integrationSettingsChanged' => $integrationSettingsChanged,
        ]);
    }

    private function getUpdateRules(Request $request): array
    {
        $rules = [
            'integration_id' => 'required|exists:integrations,id',
            'name' => 'required|unique:integration_instances,name',
            'connection_settings' => 'required|array',
            'integration_settings' => 'required|array',
            'integration_settings.*' => 'nullable', // this rule to read all options
        ];

        // Update
        if (in_array($request->method(), ['PUT', 'PATCH'])) {
            unset($rules['integration_id']);
            // TODO: Not sure how to handle the unique for updates
            $rules['name'] = 'sometimes';
            $rules['connection_settings'] = 'sometimes|array';
            $rules['integration_settings'] = 'sometimes|array';
            $rules['integration_settings.*'] = 'nullable';
        }

        return $rules;
    }

    /**
     * @throws Throwable
     */
    public function abstractDestroy(IntegrationInstanceInterface $integrationInstance)
    {
        $this->integrationInstanceRepository->delete($integrationInstance);

        return $this->response->setStatusCode(204);
    }
}