<?php

namespace App\Abstractions\Integrations\SalesChannels;

use App\DataTable\DataTableModelInterface;
use App\DataTable\DataTableModelTrait;
use App\Models\ApiLog;
use App\Models\DataImportMapping;
use App\Models\IntegrationInstance;
use App\Models\SalesOrder;
use Carbon\CarbonImmutable;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Database\Eloquent\Relations\HasOne;

/**
 * @property-read DataImportMapping $dataImportMapping
 * @property-read ApiLog[] $apiLogs
 * @property-read AbstractSalesChannelOrderLine[] $orders
 * @property-read SalesOrder[] $salesOrders
 * @property-read SalesChannelProductInterface[] $products
 */
abstract class AbstractSalesChannelIntegrationInstance extends IntegrationInstance implements DataTableModelInterface
{
    use DataTableModelTrait;

    abstract public static function getOrderClass(): string;

    abstract public static function getOrderLineClass(): string;

    abstract public static function getProductClass(): string;

    abstract public function getProductManager(): AbstractSalesChannelProductManager;

    public static function supportsUpdates(): bool
    {
        return false;
    }

    public function setDate(string $name): void
    {
        if (@$this->integration_settings[$name]) {
            // TODO: Frontend test to make sure we are passing the utc time
            $date = CarbonImmutable::parse(@$this->integration_settings[$name]);

            $settings = array_merge($this->integration_settings, [
                $name => $date->toDateTimeString(),
            ]);

            $this->integration_settings = $settings;
            $this->update();
        }
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function dataImportMapping(): HasOne
    {
        return $this->hasOne(DataImportMapping::class, 'integration_instance_id');
    }

    public function orders(): HasMany
    {
        return $this->hasMany(static::getOrderClass(), 'integration_instance_id');
    }

    public function salesOrders(): HasManyThrough
    {
        return $this->hasManyThrough(SalesOrder::class, static::getOrderClass(), 'integration_instance_id', 'sales_channel_order_id')
            ->where('sales_channel_order_type', static::getOrderClass());
    }

    public function products(): HasMany
    {
        return $this->hasMany(static::getProductClass(), 'integration_instance_id');
    }
}
