<?php

namespace App\Abstractions\Integrations\Accounting;

use App\Abstractions\Integrations\AbstractIntegrationRepository;
use App\Models\Payment;
use App\Repositories\IntegrationInstanceRepository;
use Illuminate\Contracts\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Query\JoinClause;

abstract class AbstractAccountingPaymentRepository extends AbstractIntegrationRepository implements AccountingPaymentRepositoryInterface
{
    public function getPaymentsNeedingUpdate(array $paymentIds, string $paymentType, ?int $limit = 30): Collection
    {
        $accountingPaymentTable = $this->model->getTable();

        $query = Payment::query()
            ->where('link_type', $paymentType)
            ->with([
                'accountingIntegration',
                'link.accountingTransaction.accountingIntegration',
            ])
            ->leftJoin($accountingPaymentTable, function (JoinClause $join) use ($accountingPaymentTable) {
                $join->on($accountingPaymentTable.'.id', 'payments.accounting_integration_id');
                $join->where('payments.accounting_integration_type', get_class($this->model));
            })
            ->whereNotNull('external_reference')
            ->where('payment_date', '>=', app(IntegrationInstanceRepository::class)->getAccountingInstance()->integration_settings['settings']['sync_start_date'])
            ->whereHas('link', function (Builder $query) {
                $query->whereHas('accountingTransaction', function (Builder $query) {
                    $query->whereHas('accountingIntegration', function (Builder $query) {
                        $query->whereNotNull('QboId'); //TODO: Abstract
                    });
                });
            });

        if ($paymentIds) {
            $query->whereIn('payments.id', $paymentIds);
        } else {
            $query->where(function (Builder $query) use ($accountingPaymentTable) {
                $query->whereNull('accounting_integration_id');
                $query->orWhereColumn('payments.updated_at', '>', $accountingPaymentTable.'.updated_at');
                $query->orWhereNull($accountingPaymentTable.'.updated_at');
            });
        }

        if ($limit) {
            $query->limit($limit);
        }

        return $query->get(['payments.*']);
    }

    public function getPaymentsNeedingDelete(array $payment_ids = [], ?int $limit = 30): Collection
    {
        $query = $this->model::query()
            ->leftJoin('payments', function (JoinClause $join) {
                $join->on($this->model->getTable().'.id', 'payments.accounting_integration_id');
                $join->where('payments.accounting_integration_type', $this->model);
            })
            ->whereNull('accounting_integration_id');

        if ($payment_ids) {
            $query->whereIn('payments.id', $payment_ids);
        }

        return $query->get([$this->model->getTable().'.*']);
    }

    public function linkPaymentWithAccountingPayment(Payment $skuAccountingTransaction, AbstractAccountingModel $accountingTransaction): void
    {
        $skuAccountingTransaction->accounting_integration_id = $accountingTransaction->id;
        $skuAccountingTransaction->accounting_integration_type = get_class($accountingTransaction);
        $skuAccountingTransaction->update();
    }
}
