<?php

namespace App\Abstractions\Integrations;

use App\Abstractions\Integrations\Data\IntegrationInstanceDataInterface;
use App\DTO\IntegrationInstanceDto;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;

abstract class AbstractIntegrationInstanceRepository
{
    abstract public function getIntegrationInstanceClassName(): string;

    abstract public function getIntegrationName(): string;

    protected IntegrationInstanceInterface $model;
    
    public function __construct()
    {
        $this->model = app($this->getIntegrationInstanceClassName());
    }

    public function getIntegration(): Integration
    {
        return Integration::where('name', $this->getIntegrationName())->firstOrFail();
    }

    public function all($integration_instance_id = null): EloquentCollection
    {
        $integration = $this->getIntegration();

        $query = $this->model::query();

        $query->where('integration_id', $integration->id);

        if ($integration_instance_id) {
            $query->where('id', $integration_instance_id);
        }

        return $query->get();
    }

    public function getById(int $id): IntegrationInstanceInterface
    {
        return $this->model::query()->where('id', $id)->firstOrFail();
    }

    public function save(IntegrationInstanceDataInterface $data): IntegrationInstance
    {
        return IntegrationInstance::updateOrCreate(['id' => $data->id ?? null], $data->toArray());
    }

    public function updateConnectionSettings(IntegrationInstanceInterface $integrationInstance, array $connectionSettings): void
    {
        $connectionSettings = array_merge($integrationInstance->connection_settings ?? [], $connectionSettings);
        $integrationInstance->connection_settings = $connectionSettings;
        $integrationInstance->save();
    }

    public function updateIntegrationSettings(IntegrationInstanceInterface $integrationInstance, array $integrationSettings): void
    {
        $integrationSettings = array_merge($integrationInstance->integration_settings ?? [], $integrationSettings);
        $integrationInstance->integration_settings = $integrationSettings;
        $integrationInstance->save();
    }

    public function setStatusDeleting(IntegrationInstanceInterface $integrationInstance): void
    {
        $settings = $integrationInstance->connection_settings;
        $settings['deleting'] = true;

        $integrationInstance->connection_settings = $settings;
        $integrationInstance->save();
    }

    public function delete(IntegrationInstanceInterface $integrationInstance): void
    {
        $integrationInstance->delete();
    }

    public function autoSyncExists(): bool
    {
        return $this->model::where('is_automatic_sync_enabled', true)->exists();
    }

    public function exists(): bool
    {
        return $this->model::exists();
    }

    public function allAutoSync($integration_instance_id = null): EloquentCollection
    {
        $query = $this->model::where('is_automatic_sync_enabled', true);

        if ($integration_instance_id) {
            $query->where('id', $integration_instance_id);
        }

        return $query->get();
    }

    public function getTotalCount(): int
    {
        return $this->model::count();
    }
}
