<?php

namespace Modules\Xero\Services;

use App\Exceptions\ApiException;
use App\Exceptions\UnauthorizedApiException;
use App\Models\ApiLog;
use App\Models\Concerns\WebClient;
use App\Models\IntegrationInstance;
use Carbon\Carbon;
use Exception;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Psr7\Query;
use Illuminate\Http\Client\ConnectionException;
use Illuminate\Support\Facades\Http;
use JetBrains\PhpStorm\ArrayShape;
use Modules\Xero\Collections\XeroAccountCollection;
use Modules\Xero\Collections\XeroContactCollection;
use Modules\Xero\Collections\XeroCreditNoteCollection;
use Modules\Xero\Collections\XeroInvoiceCollection;
use Modules\Xero\Collections\XeroManualJournalCollection;
use Modules\Xero\Collections\XeroPaymentCollection;
use Modules\Xero\Collections\XeroPurchaseOrderCollection;
use Modules\Xero\Collections\XeroTaxRateCollection;
use Modules\Xero\DTO\XeroAccount;
use Modules\Xero\DTO\XeroContact;
use Modules\Xero\DTO\XeroCreditNote;
use Modules\Xero\DTO\XeroInvoice;
use Modules\Xero\DTO\XeroManualJournalDto;
use Modules\Xero\DTO\XeroPayment;
use Modules\Xero\DTO\XeroPurchaseOrder;
use Modules\Xero\DTO\XeroTaxRate;
use Modules\Xero\Entities\XeroIntegrationInstance;
use Modules\Xero\Exceptions\XeroApiGeneralException;
use Modules\Xero\Exceptions\XeroDailyApiLimitReachedException;
use Modules\Xero\Exceptions\XeroReauthorizeException;
use Modules\Xero\Exceptions\XeroTimeoutException;

class Client
{
    use WebClient;

    private string $tenantId;

    private string $redirect_uri;

    private ?string $accessToken;

    private ?string $client_id;

    private ?string $client_secret;

    private array $scopes;

    private string $host;

    private string $refreshToken;

    private XeroIntegrationInstance|IntegrationInstance $xeroIntegrationInstance;

    private ?int $api_log_id = null;

    /**
     * @throws Exception
     */
    public function __construct(XeroIntegrationInstance|IntegrationInstance $integrationInstance)
    {
        $this->xeroIntegrationInstance = $integrationInstance;
        $this->refreshToken = ($this->xeroIntegrationInstance->connection_settings)['refresh_token'] ?? '';
        $this->tenantId = (($this->xeroIntegrationInstance->connection_settings)['tenant_id'] ?? '');
        $this->host = config('xero.api_host');
        $this->client_id = config('xero.oauth.client_id');
        $this->client_secret = config('xero.oauth.client_secret');
        $this->accessToken = $this->xeroIntegrationInstance->connection_settings['token'] ?? null;
        $this->scopes = config('xero.oauth.scopes');
        $this->redirect_uri = config('app.app_url') . config('xero.oauth.redirect_uri');
        if (! empty($this->refreshToken)) {
            $this->token();
        }
    }

    public function getAuthorizationUrl(): string
    {
        $baseUrl = config('xero.oauth.url_authorize').'?';
        $params = ['scope' => implode(' ', config('xero.oauth.scopes')), 'redirect_uri' => $this->redirect_uri, 'client_id' => $this->client_id, 'response_type' => 'code'];

        return $baseUrl.http_build_query($params);
    }

    /**
     * @throws ApiException
     */
    public function getAccounts(): XeroAccountCollection
    {
        $response = $this->response('GET', '/Accounts', $this->tenantId);
        $accounts = collect(json_decode($response['body'], true)['Accounts']);

        $collection = new XeroAccountCollection();

        foreach ($accounts as $account) {
            $account = XeroAccount::from([
                ...$account,
                ...['json_object' => $account],
            ]);

            $collection->add(item: $account);
        }

        return $collection;
    }

    public function getOrganization()
    {
        $response = $this->response('GET', '/Organisation', $this->tenantId);
        $organization = collect(json_decode($response['body'], true)['Organisations']);
        dd($organization->toArray());
    }

    /**
     * @param  null  $unitdp
     *
     * @throws ApiException
     * @throws Exception
     */
    #[ArrayShape(['body' => 'string', 'statusCode' => 'int', 'headers' => "\string[][]"])]
    private function response(
        string $method,
        string $endpoint,
        string $xero_tenant_id,
        ?array $body = null,
        bool $summarize_errors = false,
        $unitdp = null,
        ?array $params = [],
        array $headers = [],
        $logApiResponse = true,
    ): array {
        if (! $this->xeroIntegrationInstance->hasRemainingDailyApiCalls()) {
            throw new XeroDailyApiLimitReachedException('Xero Daily API calls limit reached');
        }

        [$method, $endpoint, $headers, $body] = $this->request($method, $endpoint, $xero_tenant_id, $body, $summarize_errors, $unitdp, $params, $headers);

        try {
            $response = Http::withHeaders($headers)
                ->timeout(30 * 4)
                ->send(
                    $method,
                    $endpoint,
                    // We are no longer encoding $body
                    //['json' => json_decode($body, true)]
                    ['json' => $body]
                );

            $headers = $response->headers();
            $body = $response->body();
            $statusCode = $response->status();
        } catch (RequestException $e) {
            $statusCode = $e->getCode();
            $headers = $e->getResponse()?->getHeaders();
            $body = $e->getResponse()?->getBody()->getContents();

            ApiLog::query()->find($this->api_log_id)->update([
                'responseStatusCode' => $statusCode,
                'responseHeaders' => $headers,
                'responseBody' => $body,
            ]);

            if ($statusCode === 403) {
                throw new UnauthorizedApiException("[$statusCode] {$e->getMessage()}", $statusCode, $e->getRequest()?->getBody(), $headers, $body);
            }

            if ($statusCode != 400) {
                throw new ApiException("[$statusCode] {$e->getMessage()}", $statusCode, $e->getRequest()?->getBody(), $headers, $body);
            }
        } catch (ConnectionException $e) {
            // Check if the error message contains 'cURL error 28'
            if (str_contains($e->getMessage(), 'cURL error 28')) {
                throw new XeroTimeoutException('Xero API timed out. Please try again later.');
            }
            throw $e;
        }

        if (isset($headers['X-DayLimit-Remaining'])) {
            $integrationSettings = $this->xeroIntegrationInstance->integration_settings;
            $integrationSettings['remainingApiUsage'] = (int) $headers['X-DayLimit-Remaining'][0];
            $retryAfter = (int) isset($headers['Retry-After']) ? ($headers['Retry-After'][0] ?? 0) : 0;
            $integrationSettings['dailyResetAt'] = Carbon::now()->addSeconds($retryAfter)->toDateTimeString();
            $this->xeroIntegrationInstance->integration_settings = $integrationSettings;
            $this->xeroIntegrationInstance->save();
        }

        if ($logApiResponse) {
            ApiLog::query()->find($this->api_log_id)->update([
                'responseStatusCode' => $statusCode,
                'responseHeaders' => $headers,
                'responseBody' => $body,
            ]);
        }

        if ($statusCode == 403) {
            throw new ApiException("[$statusCode] {$response->body()}", $statusCode, $response->body(), $headers, $body);
        }

        return ['body' => $body, 'statusCode' => $statusCode, 'headers' => $headers];
    }

    /**
     * @param  null  $unitdp
     *
     * @throws Exception
     */
    private function request(
        string $method,
        string $endpoint,
        string $xero_tenant_id,
        ?array $body = null,
        bool $summarize_errors = false,
        $unitdp = null,
        ?array $params = [],
        ?array $headers = [],
    ): array {
        $queryParams = $params;

        // Commenting this out since $body should already be in the right format
        $body = $body ?: [];

        // query params
        if ($summarize_errors !== null) {
            $queryParams['summarizeErrors'] = $summarize_errors ? 'true' : 'false';
        }
        // query params
        if ($unitdp !== null) {
            $queryParams['unitdp'] = self::toQueryValue($unitdp);
        }

        $headers = array_merge($headers, $this->getHeaders($xero_tenant_id));

        $query = Query::build($queryParams);

        /** @var ApiLog $apiLog */
        $apiLog = ApiLog::query()->create([
            'integration_instance_id' => $this->xeroIntegrationInstance->id,
            'url' => $this->host.$endpoint.($query ? "?$query" : ''),
            'requestHeaders' => $headers,
            'requestBody' => $body,
        ]);

        $this->api_log_id = $apiLog->id;

        return [
            $method,
            $this->host.$endpoint.($query ? "?$query" : ''),
            $headers,
            $body,
        ];
    }

    /**
     * @throws Exception
     */
    private function getHeaders(string $xero_tenant_id): array
    {
        $headers = $this->selectHeaders(['application/json'], ['application/json']);

        $headers['xero-tenant-id'] = $xero_tenant_id;
        if ($this->token()) {
            $headers['Authorization'] = 'Bearer '.$this->accessToken;
        }

        return $headers;
    }

    /**
     * @throws Exception
     */
    public function token(string $grant_type = 'refresh_token', ?array $code = null): string
    {
        if ($grant_type == 'refresh_token' && ($this->isAccessTokenExpired())) {
            if (empty($this->refreshToken) && config('app.env') != 'testing') {
                throw new XeroReauthorizeException($this->xeroIntegrationInstance);
            }
            $options = ['grant_type' => $grant_type, $grant_type => $this->refreshToken];
        } elseif ($grant_type == 'authorization_code') {
            $options = ['grant_type' => $grant_type, 'code' => $code['code'], 'redirect_uri' => $this->redirect_uri];
        } else {
            if (empty($this->tenantId)) {
                $this->storeTenantId($this->getTenantId($this->accessToken));
            }

            return $this->accessToken;
        }
        $response = Http::timeout(60)->withHeaders(['Content-Type' => 'application/x-www-form-urlencoded', 'Authorization' => 'Basic '.base64_encode($this->client_id.':'.$this->client_secret),

        ])->send('post', config('xero.oauth.url_access_token'), ['form_params' => $options]);

        if ($response->ok()) {
            $this->accessToken = $response->json()['access_token'];
            $this->tenantId = empty($this->tenantId) ? $this->getTenantId($this->accessToken) : $this->tenantId;
            $this->store($response->json());
        } else {
            $this->xeroIntegrationInstance->connection_settings = [
                'clientId' => $this->client_id,
                'clientSecret' => $this->client_secret,
                'tenant_id' => $this->tenantId,
            ];
            $this->xeroIntegrationInstance->save();
            throw new Exception($response->body());
        }

        return $this->accessToken;
    }

    /**
     * @throws Exception
     */
    public function isAccessTokenExpired(): bool
    {
        $expires = ($this->xeroIntegrationInstance->connection_settings)['expires'] ?? 0;

        return time() >= $expires;
    }

    /**
     * @throws Exception
     */
    public function getTenantId(string $accessToken): ?string
    {
        $response = Http::withHeaders(['Content-Type' => 'application/json', 'Authorization' => 'Bearer '.$accessToken,
        ])->get(config('xero.api_connections'));
        if ($response->ok()) {
            return $response->json()[0]['tenantId'] ?? null;
        } else {
            throw new Exception('Invalid Xero access token, can not get TenantId');
        }
    }

    public function store(array $token): void
    {
        $this->xeroIntegrationInstance->connection_settings = array_merge($this->xeroIntegrationInstance->connection_settings, ['token' => $token['access_token'], 'refresh_token' => $token['refresh_token'], 'id_token' => $token['id_token'], 'expires' => $token['expires_in'] + (time()), 'tenant_id' => $this->tenantId]);
        $this->xeroIntegrationInstance->save();
    }

    public function storeTenantId(string $tenant_id): void
    {
        $this->xeroIntegrationInstance->connection_settings = array_merge($this->xeroIntegrationInstance->connection_settings, ['tenant_id' => $tenant_id]);
        $this->xeroIntegrationInstance->save();
        $this->tenantId = $tenant_id;
    }

    /**
     * @throws ApiException
     * @throws GuzzleException
     */
    public function getTaxRates(): XeroTaxRateCollection
    {
        $response = $this->response('GET', '/TaxRates', $this->tenantId);

        $taxRates = collect(json_decode($response['body'], true)['TaxRates']);

        $collection = new XeroTaxRateCollection();

        foreach ($taxRates as $taxRate) {
            $taxRate = XeroTaxRate::from([
                ...$taxRate,
                ...['json_object' => $taxRate],
            ]);

            $collection->add(item: $taxRate);
        }

        return $collection;
    }

    /**
     * @throws ApiException
     * @throws GuzzleException
     */
    public function getInvoices(array $ids): XeroInvoiceCollection
    {
        $idsQueryParam = implode(',', $ids);

        $response = $this->response('GET', '/Invoices', $this->tenantId, null, false, 4, [
            'IDs' => $idsQueryParam,
        ]);

        $invoices = collect(json_decode($response['body'], true)['Invoices']);

        $collection = new XeroInvoiceCollection();

        foreach ($invoices as $invoice) {
            $invoice = XeroInvoice::from([
                ...$invoice,
                ...['json_object' => $invoice],
            ]);
            $collection->add(item: $invoice);
        }

        return $collection;
    }

    /**
     * @throws ApiException
     * @throws GuzzleException
     */
    public function getCreditNote(string $id): XeroCreditNoteCollection
    {
        $response = $this->response('GET', '/CreditNote/'.$id, $this->tenantId, null, false, 4);

        if (! isset($response['body'])) {
            throw new Exception('Unknown xero response: '.json_encode($response));
        }

        $responseBody = json_decode($response['body'], true);

        if (! isset($responseBody['CreditNotes'])) {
            throw new Exception('Unknown xero response for Credit Notes: '.json_encode($response));
        }

        $creditNotes = collect($responseBody['CreditNotes']);

        $collection = new XeroCreditNoteCollection();

        foreach ($creditNotes as $creditNote) {
            $creditNote = XeroCreditNote::from([
                ...$creditNote,
                ...['json_object' => $creditNote],
            ]);
            $collection->add(item: $creditNote);
        }

        return $collection;
    }

    /**
     * @throws ApiException
     * @throws Exception
     */
    public function updateOrCreateInvoices(array $invoices): XeroInvoiceCollection
    {
        $response = $this->response('POST', '/Invoices', $this->tenantId, $invoices, false, 4);

        if (! isset($response['body'])) {
            throw new Exception('Unknown xero response: '.json_encode($response));
        }

        $responseBody = json_decode($response['body'], true);

        if (! isset($responseBody['Invoices'])) {
            throw new Exception('Unknown xero response for Invoices: '.json_encode($response));
        }

        /*
         * TODO: @Bright.  We need 429 handling for all client methods.  Here is an example dd($response) when a 429 is faced
         */
        //        array:3 [ // Modules/Xero/Services/Client.php:415
        //                "body" => ""
        //          "statusCode" => 429
        //          "headers" => array:14 [
        //                "Server" => array:1 [
        //                0 => "nginx"
        //            ]
        //            "Retry-After" => array:1 [
        //                0 => "32"
        //            ]
        //            "Xero-Correlation-Id" => array:1 [
        //                0 => "fd83ed0d-0bdf-43d2-a581-fcdea716d0c0"
        //            ]
        //            "X-AppMinLimit-Remaining" => array:1 [
        //                0 => "9955"
        //            ]
        //            "X-MinLimit-Remaining" => array:1 [
        //                0 => "0"
        //            ]
        //            "X-DayLimit-Remaining" => array:1 [
        //                0 => "4774"
        //            ]
        //            "X-Rate-Limit-Problem" => array:1 [
        //                0 => "minute"
        //            ]
        //            "Content-Length" => array:1 [
        //                0 => "0"
        //            ]
        //            "Expires" => array:1 [
        //                0 => "Mon, 13 Feb 2023 20:44:17 GMT"
        //            ]
        //            "Cache-Control" => array:1 [
        //                0 => "max-age=0, no-cache, no-store"
        //            ]
        //            "Pragma" => array:1 [
        //                0 => "no-cache"
        //            ]
        //            "Date" => array:1 [
        //                0 => "Mon, 13 Feb 2023 20:44:17 GMT"
        //            ]
        //            "Connection" => array:1 [
        //                0 => "close"
        //            ]
        //            "X-Client-TLS-ver" => array:1 [
        //                0 => "tls1.3"
        //            ]
        //          ]
        //        ]

        $invoices = collect($responseBody['Invoices']);

        $collection = new XeroInvoiceCollection();

        foreach ($invoices as $invoice) {
            $invoice = XeroInvoice::from([
                ...$invoice,
                ...['json_object' => $invoice],
            ]);
            $collection->add(item: $invoice);
        }

        return $collection;
    }

    /**
     * @throws ApiException
     * @throws Exception
     */
    public function updateOrCreatePurchaseOrders(array $purchaseOrders): XeroPurchaseOrderCollection
    {
        $response = $this->response('POST', '/PurchaseOrders', $this->tenantId, $purchaseOrders, false, 4);

        $purchaseOrders = collect(json_decode($response['body'], true)['PurchaseOrders']);

        $collection = new XeroPurchaseOrderCollection();

        foreach ($purchaseOrders as $purchaseOrder) {
            $purchaseOrder = XeroPurchaseOrder::from([
                ...$purchaseOrder,
                ...['json_object' => $purchaseOrder],
            ]);
            $collection->add(item: $purchaseOrder);
        }

        return $collection;
    }

    /**
     * @throws ApiException
     * @throws GuzzleException
     */
    public function updateOrCreateCreditNotes(array $creditNotes): XeroCreditNoteCollection
    {
        $response = $this->response('POST', '/CreditNotes', $this->tenantId, $creditNotes, false, 4);

        $creditNotes = collect(json_decode($response['body'], true)['CreditNotes']);

        $collection = new XeroCreditNoteCollection();

        foreach ($creditNotes as $creditNote) {
            $creditNote = XeroCreditNote::from([
                ...$creditNote,
                ...['json_object' => $creditNote],
            ]);
            $collection->add(item: $creditNote);
        }

        return $collection;
    }

    /**
     * @throws ApiException
     * @throws GuzzleException
     */
    public function allocateCreditNotes(array $xeroAllocation, string $CreditNoteID): XeroCreditNoteCollection
    {
        $response = $this->response('PUT', '/CreditNotes/'.$CreditNoteID.'/Allocations', $this->tenantId, $xeroAllocation, false, 4);

        $creditNotes = collect(json_decode($response['body'], true)['Allocations']);
        $collection = new XeroCreditNoteCollection();

        foreach ($creditNotes as $creditNote) {
            $creditNote = XeroCreditNote::from([
                ...$creditNote,
                ...['json_object' => $creditNote],
            ]);
            $collection->add(item: $creditNote);
        }

        return $collection;
    }

    /**
     * @throws ApiException
     * @throws GuzzleException
     */
    public function updateOrCreateManualJournals(array $journals): XeroManualJournalCollection
    {
        $response = $this->response('POST', '/ManualJournals', $this->tenantId, $journals, false, 4);

        if (! isset($response['body'])) {
            throw new Exception('Unknown xero response: '.json_encode($response));
        }

        $responseBody = json_decode($response['body'], true);

        if (! isset($responseBody['ManualJournals'])) {
            throw new Exception('Unknown xero response for Manual Journals: '.json_encode($response));
        }

        $manualJournals = collect($responseBody['ManualJournals']);

        $collection = new XeroManualJournalCollection();

        foreach ($manualJournals as $manualJournal) {
            $manualJournal = XeroManualJournalDto::from([
                ...$manualJournal,
                ...['json_object' => $manualJournal],
            ]);
            $collection->add(item: $manualJournal);
        }

        return $collection;
    }

    /**
     * @throws ApiException
     */
    public function getManualJournals(?Carbon $modifiedAfter = null): XeroManualJournalCollection
    {
        $headers = [];
        if ($modifiedAfter) {
            $headers = [
                'If-Modified-Since' => $modifiedAfter->toRfc7231String(),
            ];
        }

        $response = $this->response('GET', '/ManualJournals', $this->tenantId, [], false, 4, [], $headers, false);

        $manualJournals = collect(json_decode($response['body'], true)['ManualJournals']);

        $collection = new XeroManualJournalCollection();

        foreach ($manualJournals as $manualJournal) {
            $manualJournal = XeroManualJournalDto::from([
                ...$manualJournal,
                ...['json_object' => $manualJournal],
            ]);
            $collection->add(item: $manualJournal);
        }

        return $collection;
    }

    /**
     * @throws ApiException
     * @throws Exception
     */
    public function updateOrCreateContacts(array $contacts): XeroContactCollection
    {
        $response = $this->response('POST', '/Contacts', $this->tenantId, $contacts, false, 4);

        $contacts = collect(json_decode($response['body'], true)['Contacts']);

        $collection = new XeroContactCollection();

        foreach ($contacts as $contact) {
            $contact = XeroContact::from([
                ...$contact,
                ...['json_object' => $contact],
            ]);
            $collection->add(item: $contact);
        }

        return $collection;
    }

    /**
     * @throws ApiException
     * @throws GuzzleException
     * @throws Exception
     */
    public function updateOrCreatePayments(array $payments): XeroPaymentCollection
    {
        $response = $this->response('PUT', '/Payments', $this->tenantId, $payments, false, 4);

        if (! isset($response['body'])) {
            throw new Exception('Unknown xero response: '.json_encode($response));
        }

        $responseBody = json_decode($response['body'], true);

        if (! isset($responseBody['Payments'])) {
            throw new XeroApiGeneralException('Unknown xero response for Payments: '.json_encode($response));
        }

        $payments = collect($responseBody['Payments']);

        $collection = new XeroPaymentCollection();

        foreach ($payments as $payment) {
            $payment = XeroPayment::from([
                ...$payment,
                ...['json_object' => $payment],
            ]);
            $collection->add(item: $payment);
        }

        return $collection;
    }

    /**
     * @throws ApiException
     * @throws GuzzleException
     */
    public function deletePayment(array $xeroPayment, string $PaymentID): XeroPaymentCollection
    {
        $response = $this->response('POST', '/Payments/'.$PaymentID, $this->tenantId, $xeroPayment, false, 4);

        $payments = collect(json_decode($response['body'], true)['Payments']);
        $collection = new XeroPaymentCollection();

        foreach ($payments as $payment) {
            $payment = XeroPayment::from([
                ...$payment,
                ...['json_object' => $payment],
            ]);
            $collection->add(item: $payment);
        }

        return $collection;
    }
}
