<?php

namespace Modules\Xero\Repositories;

use App\Exceptions\ApiException;
use App\Helpers;
use App\Models\Payment;
use App\Repositories\IntegrationInstanceRepository;
use Exception;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Xero\DTO\XeroPayment as XeroPaymentDTO;
use Modules\Xero\DTO\XeroPayments;
use Modules\Xero\Entities\XeroPayment;
use Modules\Xero\Entities\XeroTransaction;
use Modules\Xero\Services\Client;

class XeroPaymentRepository
{
    public function getPaymentsNeedingUpdate(array $payment_ids = []): EloquentCollection
    {
        $query = Payment::query()
            ->leftJoin('xero_payments', function (JoinClause $join) {
                $join->on('xero_payments.id', 'payments.accounting_integration_id');
                $join->where('payments.accounting_integration_type', XeroPayment::class);
            })
            ->where('payment_date', '>=', app(IntegrationInstanceRepository::class)->getAccountingInstance()->integration_settings['settings']['sync_start_date'])
            ->whereHas('link', function (Builder $query) {
                $query->whereHas('accountingTransaction', function (Builder $query) {
                    $query->whereHas('accountingIntegration', function (Builder $query) {
                        $query->whereNotNull('xero_uuid');
                    });
                });
            });

        if ($payment_ids) {
            $query->whereIn('payments.id', $payment_ids);
        } else {
            $query->where(function (Builder $query) {
                $query->whereNull('accounting_integration_id');
                $query->orWhereColumn('payments.updated_at', '>', 'xero_payments.updated_at');
                $query->orWhereNull('xero_payments.updated_at');
            });
        }

        customlog('xero', 'Syncing '.$query->count().' Xero payments');

        return $query->get(['payments.*']);
    }

    public function getPaymentsNeedingDelete(array $payment_ids = []): EloquentCollection
    {
        $query = XeroPayment::query()
            ->leftJoin('payments', function (JoinClause $join) {
                $join->on('xero_payments.id', 'payments.accounting_integration_id');
                $join->where('payments.accounting_integration_type', XeroPayment::class);
            })
            ->whereNull('accounting_integration_id')
            ->whereNotNull('PaymentID');

        if ($payment_ids) {
            $query->whereIn('payments.id', $payment_ids);
        }

        return $query->get(['xero_payments.*']);
    }

    /**
     * @throws Exception
     */
    public function deleteXeroPayments(Collection $payments)
    {
        if (! $payments->count()) {
            return;
        }
        $payments->whereNotNull('PaymentID')->each(function (XeroPayment $xeroPayment) {

            $xeroPaymentDTO = XeroPaymentDTO::from([
                'Status' => XeroPaymentDTO::STATUS_DELETED,
            ]);

            $client = new Client(app(IntegrationInstanceRepository::class)->getAccountingInstance());

            try {
                $responsePayments = $client->deletePayment($xeroPaymentDTO->toArray(), $xeroPayment->PaymentID);
            } catch (GuzzleException|ApiException) {
                $xeroPayment->last_error = Arr::wrap('Connection error');
                $xeroPayment->save();

                return;
            }

            /** @var XeroPaymentDTO $responsePayment */
            foreach ($responsePayments as $responsePayment) {
                DB::transaction(function () use ($responsePayment, $xeroPayment) {

                    $has_error = ! empty((array) $responsePayment->ValidationErrors);

                    if ($has_error) {
                        if ($xeroPayment->json_object['Status'] == XeroPaymentDTO::STATUS_DELETED) {
                            $xeroPayment->delete();
                        } else {
                            $xeroPayment->last_error = array_unique(Arr::flatten($responsePayment->ValidationErrors));
                            $xeroPayment->json_object = $responsePayment->json_object;
                            $xeroPayment->save();
                        }
                    } else {
                        $xeroPayment->delete();
                    }
                });
            }
        });
    }

    /**
     * @throws Exception
     */
    private function updateOrCreatePayments(Collection $payments)
    {
        if (! $payments->count()) {
            return;
        }
        $xeroPayments = [];
        $paymentsMap = [];

        $payments->each(function (Payment $payment) use (&$xeroPayments, &$paymentsMap) {
            $accountingTransaction = $payment->link->accountingTransaction;
            $bankAccountID = $payment->paymentType->accountingIntegration?->AccountID;

            /** @var XeroTransaction $xeroTransaction */
            $xeroTransaction = $accountingTransaction->accountingIntegration;

            if (! $xeroTransaction || ! $bankAccountID) {
                return;
            }

            $invoiceArray = [
                'InvoiceID' => $xeroTransaction->xero_uuid,
            ];

            $accountArray = [
                'AccountID' => $bankAccountID,
            ];

            $xeroPayments[] = XeroPaymentDTO::from([
                'Invoice' => $invoiceArray,
                'Account' => $accountArray,
                'Date' => Helpers::dateUtcToLocal($payment->payment_date)->format('Y-m-d'),
                'Amount' => abs($payment->amount),
                'Reference' => $payment->external_reference,
                'CurrencyRate' => $payment->currency_rate == 0 ? 0 : (1 / $payment->currency_rate),
            ]);
            $paymentsMap[] = $payment->id;
        });

        if (empty($xeroPayments)) {
            return;
        }

        $requestPayments = XeroPayments::from([
            'Payments' => $xeroPayments,
        ]);

        $client = new Client(app(IntegrationInstanceRepository::class)->getAccountingInstance());

        try {
            $responsePayments = $client->updateOrCreatePayments($requestPayments->toArray());
        } catch (GuzzleException|ApiException) {
            $payments->each(function (Payment $payment) {
                /** @var XeroPayment $xeroPayment */
                $xeroPayment = XeroPayment::query()->firstOrNew([
                    'id' => $payment->accounting_integration_id ?? null,
                ]);
                $xeroPayment->last_error = Arr::wrap('Connection error');
                $xeroPayment->timestamps = false;
                $xeroPayment->save();

                $payment->accounting_integration_id = $xeroPayment->id;
                $payment->accounting_integration_type = XeroPayment::class;
                $payment->timestamps = false;
                $payment->save();
            });

            return;
        }

        /** @var XeroPayment $xeroPayment */
        foreach ($responsePayments as $xeroPaymentDTO) {
            DB::transaction(function () use ($xeroPaymentDTO, &$paymentsMap) {

                $has_error = ($xeroPaymentDTO->StatusAttributeString && $xeroPaymentDTO->StatusAttributeString != 'OK') || ! empty((array) $xeroPaymentDTO->ValidationErrors);
                /** @var Payment $payment */
                $payment = Payment::query()->findOrFail(array_shift($paymentsMap));

                /** @var XeroPayment $xeroPayment */
                $xeroPayment = XeroPayment::query()->firstOrNew([
                    'id' => $payment->accounting_integration_id ?? null,
                ]);

                if ($has_error) {
                    $xeroPayment->last_error = Arr::wrap(array_unique(Arr::flatten($xeroPaymentDTO->ValidationErrors)));
                    $xeroPayment->timestamps = false;
                    if (! $payment->accounting_integration_id) {
                        $xeroPayment->created_at = now();
                    }
                } else {
                    $xeroPayment->PaymentID = $xeroPaymentDTO->PaymentID;
                    $xeroPayment->last_error = null;
                    $xeroPayment->json_object = $xeroPaymentDTO->json_object;
                    /*
                     * The purpose of the following was to update the invoice payload in our xero_transactions table
                     * after payment is made (A payment made modifies the payload of a transaction).  However, what
                     * is happening is that the payload being returned is being associated wrong. The following query
                     * identifies such issues:
                     *
                     * SELECT xt.id, JSON_EXTRACT(xt.json_object, "$.CreditNoteID"), xt.json_object, xt.updated_at, xt.created_at, at.id
                     * FROM xero_transactions AS xt INNER JOIN accounting_transactions AS at
                     * ON at.accounting_integration_id = xt.id AND at.accounting_integration_type = "Modules\\Xero\\Entities\\XeroTransaction"
                     * WHERE xt.`type`="ACCRECCREDIT" AND JSON_EXTRACT(xt.json_object, "$.CreditNoteID") IS NULL
                     */
                    /*if ($invoice = $xeroPaymentDTO->Invoice)
                    {
                        if ($transaction = XeroTransaction::where('xero_uuid', $invoice['InvoiceID'])->first())
                        {
                            $transaction->last_error = null;
                            $transaction->json_object = $invoice;
                            $transaction->update();
                        }
                    }*/
                }
                $xeroPayment->save();

                if (! $payment->accounting_integration_id) {
                    $payment->accounting_integration_id = $xeroPayment->id;
                    $payment->accounting_integration_type = XeroPayment::class;
                    $payment->timestamps = false;
                    $payment->save();
                }
            });
        }

    }

    public function getFromPaymentIds(array $ids = []): EloquentCollection
    {
        return XeroPayment::query()
            ->whereHas('payment', function (Builder $query) use ($ids) {
                $query->whereIn('id', $ids);
            })
            ->get();
    }

    public function updateOrCreateXeroPayments(Collection $payments): void
    {
        $this->updateOrCreatePayments($payments);
    }
}
