<?php

namespace Modules\Xero\Repositories;

use App\Exceptions\ApiException;
use App\Models\Supplier;
use App\Repositories\IntegrationInstanceRepository;
use Carbon\Carbon;
use Exception;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Xero\Collections\XeroContactCollection;
use Modules\Xero\DTO\XeroAddress;
use Modules\Xero\DTO\XeroContact as XeroContactDTO;
use Modules\Xero\DTO\XeroContacts;
use Modules\Xero\DTO\XeroPhone;
use Modules\Xero\Entities\XeroContact;
use Modules\Xero\Services\Client;

class XeroContactRepository
{
    private IntegrationInstanceRepository $integrationInstanceRepository;

    public function __construct()
    {
        $this->integrationInstanceRepository = app(IntegrationInstanceRepository::class);
    }

    /**
     * @param  array  $customers
     */
    /*public function getCustomersNeedingUpdate(array $customers = []): \Illuminate\Database\Eloquent\Collection
    {
        $query = Customer::query()
            ->leftJoin('xero_contacts', function (JoinClause $join) {
                $join->on('xero_contacts.link_id', 'customers.id');
                $join->where('xero_contacts.link_type', Customer::class);
            })
            ->where(function (Builder $query) {
                $query->whereNull('xero_contacts.link_id');
                $query->orWhereColumn('customers.updated_at', '>', 'xero_contacts.updated_at');
                $query->orWhereNull('xero_contacts.updated_at');

            });
        if ($customers)
        {
            $query->where(function (Builder $customerQuery) use ($customers) {
                foreach ($customers as $customer) {
                    $customerQuery->orWhere('customers.id', $customer['id']);
                }
            });
        }

        return $query->get(['customers.*']);
    }*/

    public function getSuppliersNeedingUpdate(array $supplier_ids = []): \Illuminate\Database\Eloquent\Collection
    {
        $query = Supplier::query()
            ->leftJoin('xero_contacts', function (JoinClause $join) {
                $join->on('xero_contacts.link_id', 'suppliers.id');
                $join->where('xero_contacts.link_type', Supplier::class);
            })
            ->where(function (Builder $query) {
                $query->whereNull('xero_contacts.link_id');
                $query->orWhereColumn('suppliers.updated_at', '>', 'xero_contacts.updated_at');
                $query->orWhereNull('xero_contacts.updated_at');

            });
        if ($supplier_ids) {
            $query->where(function (Builder $supplierQuery) use ($supplier_ids) {
                $supplierQuery->orWhereIn('suppliers.id', $supplier_ids);
            });
        }

        return $query->get(['suppliers.*']);
    }

    public function updateXeroSuppliers(XeroContactCollection $suppliers, array $suppliersMap): void
    {
        /** @var XeroContactDTO $xeroContact */
        foreach ($suppliers as $xeroContactDTO) {
            DB::transaction(function () use ($xeroContactDTO, &$suppliersMap) {

                $has_error = $xeroContactDTO->StatusAttributeString && $xeroContactDTO->StatusAttributeString != 'OK';

                /** @var Supplier $supplier */
                $supplier = Supplier::query()->findOrFail(array_shift($suppliersMap));

                /** @var XeroContact $xeroContact */
                $xeroContact = XeroContact::query()->firstOrNew([
                    'link_id' => $supplier->id,
                    'link_type' => Supplier::class,
                ]);

                if ($has_error) {
                    $xeroContact->last_error = json_encode(array_unique(Arr::flatten($xeroContactDTO->ValidationErrors)));
                    $xeroContact->timestamps = false;
                    if (! $xeroContact->link_id) {
                        $xeroContact->created_at = now();
                    }
                } else {
                    $xeroContact->last_error = null;
                    $xeroContact->json_object = json_encode($xeroContactDTO->json_object);
                }
                $xeroContact->save();
            });
        }
    }

    /**
     * @throws Exception
     */
    private function updateOrCreateSuppliers(Collection $suppliers)
    {
        if (! $suppliers->count()) {
            return;
        }
        $xeroContacts = [];
        $suppliersMap = [];

        $suppliers->each(function (Supplier $supplier) use (&$xeroContacts, &$suppliersMap) {

            $xeroAddress = XeroAddress::from([
                XeroAddress::ADDRESS_TYPE_BILLING,
                $supplier->address?->address1,
                $supplier->address?->address2,
                $supplier->address?->address3,
                $supplier->address?->city,
                $supplier->address?->zip,
                $supplier->address?->country,
                $supplier->address?->name,
            ]);

            $xeroPhone = XeroPhone::from([
                'PhoneType' => XeroPhone::PHONE_TYPE_DEFAULT,
                'PhoneNumber' => $supplier->phone,
            ]);

            $xeroContacts[] = XeroContactDTO::from([
                'Name' => $supplier->name,
                'EmailAddress' => $supplier->email,
                'Addresses' => Arr::wrap($xeroAddress),
                'Phones' => Arr::wrap($xeroPhone),
            ]);
            $suppliersMap[] = $supplier->id;
        });

        $requestContacts = XeroContacts::from([
            'Contacts' => $xeroContacts,
        ]);

        $client = new Client(app(IntegrationInstanceRepository::class)->getAccountingInstance());

        try {
            $responseContacts = $client->updateOrCreateContacts($requestContacts->toArray());
        } catch (GuzzleException|ApiException) {
            $suppliers->each(function (Supplier $supplier) {
                /** @var XeroContact $xeroContact */
                $xeroContact = XeroContact::query()->firstOrNew([
                    'link_id' => $supplier->id,
                    'link_type' => Supplier::class,
                ]);
                $xeroContact->last_error = Arr::wrap('Connection error');
                $xeroContact->timestamps = false;
                $xeroContact->save();
            });

            return;
        }

        /** @var XeroContactDTO $xeroContact */
        foreach ($responseContacts as $xeroContactDTO) {
            DB::transaction(function () use ($xeroContactDTO, &$suppliersMap) {

                $has_error = $xeroContactDTO->StatusAttributeString && $xeroContactDTO->StatusAttributeString != 'OK';

                /** @var Supplier $supplier */
                $supplier = Supplier::query()->findOrFail(array_shift($suppliersMap));

                /** @var XeroContact $xeroContact */
                $xeroContact = XeroContact::query()->firstOrNew([
                    'link_id' => $supplier->id,
                    'link_type' => Supplier::class,
                ]);

                if ($has_error) {
                    $xeroContact->last_error = json_encode(array_unique(Arr::flatten($xeroContactDTO->ValidationErrors)));
                    $xeroContact->timestamps = false;
                    if (! $xeroContact->link_id) {
                        $xeroContact->created_at = now();
                    }
                } else {
                    $xeroContact->last_error = null;
                    $xeroContact->json_object = json_encode($xeroContactDTO->json_object);
                }
                $xeroContact->save();
            });
        }

        $integrationInstance = $this->integrationInstanceRepository->getAccountingInstance();
        $settings = $integrationInstance->integration_settings;
        $settings['lastSyncContacts'] = Carbon::now();
        $integrationInstance->integration_settings = $settings;
        $integrationInstance->update();
    }

    /**
     * @throws Exception
     */
    public function getContactIDFromSupplier(Supplier $supplier): string
    {
        /** @var XeroContact $xeroContact */
        if (! $xeroContact = XeroContact::query()->where('Name', $supplier->name)->first()) {
            $this->updateOrCreateSuppliers(Collect(Arr::wrap($supplier)));

            /** @var XeroContact $xeroContact */
            $xeroContact = XeroContact::query()->where('Name', $supplier->name)->firstOrFail();
        }

        return $xeroContact->ContactID;
    }

    public function syncContact(array $payload): XeroContact
    {
        /** @var XeroContact $xeroContact */
        $xeroContact = XeroContact::query()->firstOrNew([
            'link_id' => $payload['link_id'],
            'link_type' => $payload['link_type'],
        ]);
        $xeroContact->last_error = $payload['last_error'];
        $xeroContact->timestamps = false;
        $xeroContact->save();

        return $xeroContact;
    }
}
