<?php

namespace Modules\Xero\Entities;

use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use App\Models\Payment;
use App\Services\Accounting\AccountingIntegration;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\MorphOne;

/**
 * @property int $id
 * @property int $PaymentID
 * @property string $type,
 * @property ?array $last_error,
 * @property array $json_object
 * @property-read string $link
 * @property-read Payment $payment
 * @property Carbon $last_synced_at
 * @property Carbon $updated_at
 * @property Carbon $created_at
 */
class XeroPayment extends Model implements AccountingIntegration, Filterable
{
    use HasFactory;
    use HasFilters;

    protected $fillable = [
        'last_error',
        'json_object',
    ];

    protected $casts = [
        'last_error' => 'array',
        'json_object' => 'array',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function payment(): MorphOne
    {
        return $this->morphOne(Payment::class, 'accounting_integration');
    }

    public function link(): Attribute
    {
        return Attribute::get(fn () => 'https://go.xero.com/Bank/ViewTransaction.aspx?bankTransactionID='.$this->PaymentID);
    }

    public function lastSyncedAt(): Attribute
    {
        return Attribute::get(fn () => $this->updated_at);
    }

    public function getLastSyncedAt(): ?Carbon
    {
        return $this->last_synced_at;
    }

    public function getLastError(): ?array
    {
        return $this->last_error;
    }

    public function getLink(): ?string
    {
        return $this->link;
    }

    public function delete(): ?bool
    {
        $this->payment?->update([
            'accounting_integration_type' => null,
            'accounting_integration_id' => null,
        ]);

        return parent::delete();
    }

    public function availableColumns()
    {
        return ['id', 'PaymentID', 'type', 'last_error', 'json_object', 'created_at', 'updated_at'];
    }

    public function filterableColumns(): array
    {
        return ['id', 'PaymentID', 'type', 'last_error', 'json_object', 'created_at', 'updated_at'];
    }

    public function generalFilterableColumns(): array
    {
        return [];
    }
}
