<?php

namespace Feature\Managers;

use App\Exceptions\SalesOrder\SalesOrderFulfillmentDispatchException;
use App\Models\Address;
use App\Models\Customer;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\Supplier;
use App\Models\SupplierInventory;
use App\Models\Warehouse;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use Exception;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Facades\Queue;
use Modules\WooCommerce\Entities\WooCommerceIntegrationInstance;
use Modules\WooCommerce\Entities\WooCommerceOrder;
use Modules\WooCommerce\Entities\WooCommerceOrderItem;
use Modules\WooCommerce\Entities\WooCommerceProduct;
use Modules\WooCommerce\Enums\WooCommerceOrderStatusEnum;
use Modules\WooCommerce\Jobs\WooCommerceFulfillmentJob;
use Modules\WooCommerce\Managers\WooCommerceOrderManager;
use Modules\WooCommerce\Tests\WooCommerceMockRequests;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class WooCommerceOrderManagerTest extends TestCase
{
    use FastRefreshDatabase;
    use WooCommerceMockRequests;

    private WooCommerceIntegrationInstance $integrationInstance;
    private WooCommerceOrderManager $wooCommerceOrderManager;

    /**
     * @throws Exception
     */
    public function setUp(): void
    {
        parent::setUp();
        $this->integrationInstance = WooCommerceIntegrationInstance::factory([
            'is_automatic_sync_enabled' => true,
        ])
            ->has(SalesChannel::factory())
            ->create();

        $this->wooCommerceOrderManager = new WooCommerceOrderManager($this->integrationInstance);

        Queue::fake();
    }

    /**
     * @throws Throwable
     */
    public function test_woo_commerce_handle_cancel_sales_orders()
    {
        $wooCommerceOrder = WooCommerceOrder::factory()
            ->for($this->integrationInstance)
            ->hasOrderItems()
            ->create()
            ->refresh();
        $wooCommerceOrder->json_object = array_merge($wooCommerceOrder->json_object, [
            'status' => WooCommerceOrderStatusEnum::COMPLETED->value,
        ]);
        $wooCommerceOrder->update();

        //Create SKU orders
        $this->wooCommerceOrderManager->createSkuOrders([], true);
        $wooCommerceOrder->load('salesOrder');

        //order canceled
        $wooCommerceOrder->json_object = array_merge($wooCommerceOrder->json_object, [
            'status' => WooCommerceOrderStatusEnum::CANCELLED->value,
        ]);
        $wooCommerceOrder->update();

        //Order should not be canceled
        $this->assertTrue(is_null($wooCommerceOrder->salesOrder->canceled_at));

        //Amazon Order is canceled
        $this->wooCommerceOrderManager->handleCancelSalesOrders(collect([
            [
                'id' => $wooCommerceOrder->woo_commerce_id,
            ],
        ]));
        $wooCommerceOrder->salesOrder->refresh();

        //Sales order must be canceled
        $this->assertFalse(is_null($wooCommerceOrder->salesOrder->canceled_at));
    }

    /**
     * @throws SalesOrderFulfillmentDispatchException
     * @throws BindingResolutionException
     * @throws Throwable
     */
    public function test_woo_commerce_fulfillment_gets_submitted_on_fulfill(): void
    {
        $warehouse = Warehouse::first();
        $product = Product::factory()->create();

        $product->setInitialInventory($warehouse->id, 10);

        $wooCommerceOrder = WooCommerceOrder::factory([
            'integration_instance_id' => $this->integrationInstance->id
        ])
            ->hasOrderItems()
            ->create();

        $salesOrder = SalesOrder::factory([
            'sales_channel_id' => $this->integrationInstance->salesChannel->id,
            'sales_order_number' => $wooCommerceOrder->orderId,
            'sales_channel_order_id' => $wooCommerceOrder->id,
            'sales_channel_order_type' => WooCommerceOrder::class
        ])
            ->has(SalesOrderLine::factory()->state([
                'sales_channel_line_id' => $wooCommerceOrder->orderItems->first()->lineItemId,
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
            ]))
            ->create();

        $fulfillmentDate = now();

        app(FulfillmentManager::class)->fulfill($salesOrder, [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'warehouse_id' => $warehouse->id,
            'fulfilled_at' => $fulfillmentDate,
            'fulfilled_shipping_method' => 'USPS',
            'tracking_number' => '123456789',
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrder->salesOrderLines->first()->id,
                    'quantity' => 1,
                ]
            ],
        ], false);

        Queue::assertPushed(WooCommerceFulfillmentJob::class);
    }

    /**
     * @throws Throwable
     */
    public function test_can_assign_dropship_warehouse_to_woo_commerce_sales_order()
    {
        $address = Address::factory()->create();

        $customer = Customer::factory([
            'default_shipping_address_id' => $address->id,
        ])
            ->create();

        $product = Product::factory([
            'sku' => 'ABC'
        ])->create();

        $supplier = Supplier::factory()
            ->hasSupplierProducts(1, [
                'product_id' => $product->id,
                'is_default' => true,
            ])
            ->create();

        $warehouse = Warehouse::factory([
            'type' => Warehouse::TYPE_SUPPLIER,
            'supplier_id' => $supplier->id,
            'dropship_enabled' => true,
        ])->create();

        $supplierInventory = SupplierInventory::first();
        $supplierInventory->in_stock = true;
        $supplierInventory->warehouse_id = $warehouse->id;
        $supplierInventory->update();
        $supplierInventory->refresh();

        // Create warehouse priority
        Setting::query()->updateOrCreate(['key' => Setting::KEY_WAREHOUSE_PRIORITY], [
            'key' => Setting::KEY_WAREHOUSE_PRIORITY,
            'value' => json_encode([Warehouse::first()->id, $warehouse->id]),
        ]);
        Setting::query()->updateOrCreate(['key' => Setting::KEY_WAREHOUSE_IGNORE_SUPPLIER_TYPE], [
            'key' => Setting::KEY_WAREHOUSE_IGNORE_SUPPLIER_TYPE,
            'value' => false,
        ]);

        $supplier->default_warehouse_id = $warehouse->id;
        $supplier->update();
        $supplier->refresh();

        $wooCommerceProduct = WooCommerceProduct::factory([
            'integration_instance_id' => $this->integrationInstance->id,
        ])
            ->create();
        $wooCommerceProduct->json_object = array_merge($wooCommerceProduct->json_object, [
            'sku' => $product->sku,
        ]);
        $wooCommerceProduct->update();
        $wooCommerceProduct->refresh();

        ProductListing::factory([
            'product_id' => $product->id,
            'sales_channel_id' => $this->integrationInstance->salesChannel->id,
            'document_id' => $wooCommerceProduct->id,
            'document_type' => WooCommerceProduct::class,
            'listing_sku' => $product->sku,
        ])
            ->create();

        $wooCommerceOrder = WooCommerceOrder::factory()
            ->for($this->integrationInstance)
            ->hasOrderItems()
            ->create();

        $wooCommerceOrderItem = $wooCommerceOrder->orderItems->first();
        $wooCommerceOrderItem->json_object = array_merge($wooCommerceOrderItem->json_object, [
            'product_id' => $wooCommerceProduct->woo_commerce_id,
            'sku' => $product->sku,
        ]);
        $wooCommerceOrderItem->update();
        $wooCommerceOrderItem->refresh();

        (new WooCommerceOrderManager($this->integrationInstance))->createSkuOrders([$wooCommerceOrder->id], false, false);

        $this->assertDatabaseHas((new Product())->getTable(), [
            'sku' => $product->sku,
        ]);
        $this->assertDatabaseHas((new WooCommerceProduct())->getTable(), [
            'sku' => $product->sku,
        ]);
        $this->assertDatabaseHas((new WooCommerceOrderItem())->getTable(), [
            'product_id' => $wooCommerceProduct->woo_commerce_id,
        ]);
        $this->assertDatabaseHas((new ProductListing())->getTable(), [
            'listing_sku' => $product->sku,
            'document_id' => $wooCommerceProduct->id,
            'document_type' => WooCommerceProduct::class,
        ]);
        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
        ]);
        $this->assertDatabaseEmpty((new InventoryMovement())->getTable());
    }

    public function test_it_can_open_reserved_order_from_woo_commerce(): void
    {

    }
}
