<?php

namespace Modules\WooCommerce\Tests\Feature\Controllers;

use App\Data\CreateSkuOrderFromSalesChannelData;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\User;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Modules\WooCommerce\ApiDataTransferObjects\WooCommerceGetOrdersAdt;
use Modules\WooCommerce\Entities\WooCommerceIntegrationInstance;
use Modules\WooCommerce\Entities\WooCommerceOrder;
use Modules\WooCommerce\Entities\WooCommerceOrderItem;
use Modules\WooCommerce\Jobs\RefreshWooCommerceOrdersJob;
use Modules\WooCommerce\Managers\WooCommerceOrderManager;
use Modules\WooCommerce\Tests\WooCommerceMockRequests;
use Modules\WooCommerce\Tests\WooCommerceTestingData;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class WooCommerceOrderControllerTest extends TestCase
{
    use FastRefreshDatabase;
    use WooCommerceMockRequests;

    private WooCommerceIntegrationInstance $wooCommerceIntegrationInstance;

    public function setUp(): void
    {
        parent::setUp();
        $this->wooCommerceIntegrationInstance = WooCommerceIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create();

        Queue::fake();
        Sanctum::actingAs(User::first());
    }

    public function test_woo_commerce_get_order(): void
    {
        /*
        |--------------------------------------------------------------------------
        | Refresh order
        |--------------------------------------------------------------------------
        */

        $this->mockRefreshOrder();

        $this->postJson(route('woo-commerce.orders.refresh-order', $this->wooCommerceIntegrationInstance->id), [
            'uniqueId' => (string) WooCommerceTestingData::getOrder()['id'],
        ])->assertOk();

        $this->assertDatabaseHas((new WooCommerceOrder())->getTable(), [
            'integration_instance_id' => $this->wooCommerceIntegrationInstance->id,
            'json_object' => json_encode(WooCommerceTestingData::getOrder()),
        ]);

        $this->assertDatabaseHas((new WooCommerceOrderItem())->getTable(), [
            'json_object' => json_encode(WooCommerceTestingData::getOrder()['line_items'][0]),
        ]);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_woo_commerce_orders_controller(): void
    {
        /*
        |--------------------------------------------------------------------------
        | Refresh orders
        |--------------------------------------------------------------------------
        */

        $this->mockRefreshOrders();

        $this->postJson(route('woo-commerce.orders.refresh', $this->wooCommerceIntegrationInstance->id))->assertOk();

        Queue::assertPushed(RefreshWooCommerceOrdersJob::class);

        (new WooCommerceOrderManager($this->wooCommerceIntegrationInstance))->refreshOrders(new WooCommerceGetOrdersAdt());

        $this->assertDatabaseHas((new WooCommerceOrder())->getTable(), [
            'integration_instance_id' => $this->wooCommerceIntegrationInstance->id,
            'json_object' => json_encode(WooCommerceTestingData::getOrder()),
        ]);

        $this->assertDatabaseHas((new WooCommerceOrderItem())->getTable(), [
            'json_object' => json_encode(WooCommerceTestingData::getOrder()['line_items'][0]),
        ]);

        /*
        |--------------------------------------------------------------------------
        | Get orders
        |--------------------------------------------------------------------------
        */

        $response = $this->getJson(route('woo-commerce.orders.index', $this->wooCommerceIntegrationInstance->id))->assertOk();

        $response->assertJsonStructure([
            'data' => [
                '*' => [
                    'id',
                    'sku_sales_order',
                    'woo_commerce_id',
                    'number',
                    'status',
                    'error_log',
                    'currency',
                    'date_created_gmt',
                    'date_modified_gmt',
                    'total',
                    'total_tax',
                    'created_at',
                    'updated_at',
                ],
            ],
        ]);

        $firstOrderId = $response->json('data')[0]['id'];

        /*
        |--------------------------------------------------------------------------
        | Show order
        |--------------------------------------------------------------------------
        */

        $response = $this->getJson(route('woo-commerce.orders.show', [$this->wooCommerceIntegrationInstance->id, $firstOrderId]))->assertOk();

        $response->assertJsonStructure([
            'data' => [
                'id',
                'integration_instance_id',
                'sku_sales_order',
                'woo_commerce_id',
                'number',
                'status',
                'error_log',
                'currency',
                'date_created_gmt',
                'date_modified_gmt',
                'total',
                'total_tax',
                'json_object',
                'created_at',
                'updated_at',
            ],
        ]);

        /*
        |--------------------------------------------------------------------------
        | Create sku orders
        |--------------------------------------------------------------------------
        */

        $this->postJson(route('woo-commerce.orders.create-sku-orders', $this->wooCommerceIntegrationInstance->id), CreateSkuOrderFromSalesChannelData::from([
            'create_all_orders' => true,
        ])->toArray())->assertOk();

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'sales_channel_id' => $this->wooCommerceIntegrationInstance->salesChannel->id,
            'sales_order_number' => WooCommerceTestingData::getOrder()['number'],
            'sales_channel_order_type' => WooCommerceOrder::class,
            'tax_total' => 6.00,
        ]);

        $this->assertDatabaseCount((new SalesOrderLine())->getTable(), 2);

        $this->assertEquals(2, SalesOrderLine::whereNotNull('tax_rate_id')->count());

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'order_status' => 'reserved',
        ]);

        // Complete status - move sales order from reserved to open

        $salesChannelOrder = WooCommerceOrder::first();
        $jsonObject = $salesChannelOrder->json_object;
        $jsonObject['status'] = 'completed';
        $salesChannelOrder->json_object = $jsonObject;
        $salesChannelOrder->updated_at = now()->addMinute();
        $salesChannelOrder->save();

        (new WooCommerceOrderManager($this->wooCommerceIntegrationInstance))->updateSkuOrders(updateForAll: true);
        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'order_status' => 'open',
        ]);

        $needingUpdate = (new WooCommerceOrderManager($this->wooCommerceIntegrationInstance))->getSalesChannelOrdersForUpdate(createForAll: true);
        $this->assertCount(0, $needingUpdate);

        /*
        |--------------------------------------------------------------------------
        | Export orders
        |--------------------------------------------------------------------------
        */

        $this->getJson(route('woo-commerce.orders.export', $this->wooCommerceIntegrationInstance->id))->assertOk()->assertDownload();
    }
}
