<?php

namespace Modules\WooCommerce\Tests\Feature\Controllers;

use App\Abstractions\Integrations\Data\SalesChannelIntegrationSettingsData;
use App\Data\SalesChannelInventorySettingsData;
use App\Data\SalesChannelMasterOfPriceSettingsData;
use App\Data\SalesChannelPricingSettingsData;
use App\Jobs\CacheProductListingPriceJob;
use App\Jobs\GenerateCacheProductListingQuantityJob;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\Store;
use App\Models\User;
use Carbon\Carbon;
use Exception;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Modules\WooCommerce\Data\StoreWooCommerceIntegrationData;
use Modules\WooCommerce\Data\UpdateWooCommerceIntegrationData;
use Modules\WooCommerce\Data\WooCommerceAuthorizationResponseData;
use Modules\WooCommerce\Data\WooCommerceConnectionSettingsData;
use Modules\WooCommerce\Data\WooCommerceIntegrationSettingsData;
use Modules\WooCommerce\Entities\WooCommerceIntegrationInstance;
use Modules\WooCommerce\Entities\WooCommerceOrder;
use Modules\WooCommerce\Entities\WooCommerceOrderItem;
use Modules\WooCommerce\Entities\WooCommerceProduct;
use Modules\WooCommerce\Jobs\RefreshWooCommerceOrdersJob;
use Modules\WooCommerce\Jobs\RefreshWooCommerceProductsJob;
use Modules\WooCommerce\Services\WooCommerceClient;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class WooCommerceIntegrationInstanceControllerTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    public function setUp(): void
    {
        parent::setUp();

        Queue::fake();
        Sanctum::actingAs(User::first());
    }

    /**
     * @throws Exception
     */
    public function test_woocommerce_integration_instance_controller(): void
    {
        /*
        |--------------------------------------------------------------------------
        | Create integration instance
        |--------------------------------------------------------------------------
        */

        $integration = Integration::where('name', Integration::NAME_WOOCOMMERCE)->first();

        $response = $this->postJson(route('woo-commerce.store'), StoreWooCommerceIntegrationData::from([
            'name' => 'WooCommerce',
            'integration_id' => $integration->id,
            'connection_settings' => WooCommerceConnectionSettingsData::from([]),
            'integration_settings' => WooCommerceIntegrationSettingsData::from([
                'url' => 'https://www.store.com',
                'start_date' => Carbon::now()->subYear()->toDateTimeString(),
                'store_id' => Store::first()->id,
                'auto_link_products' => false,
                'auto_create_products' => false,
                'sales_nominal_code_id' => null,
                'cogs_nominal_code_id' => null,
                'pricing' => SalesChannelPricingSettingsData::from([
                    'masterOfPrice' => SalesChannelMasterOfPriceSettingsData::from([])
                ]),
                'inventory' => SalesChannelInventorySettingsData::from([]),
                'emailCustomers' => false,
                'proforma_marketplace_cost_percentage' => null,
                'proforma_payment_cost_percentage' => null,
                'sync_sales_order_invoices_to_accounting' => true
            ]),
        ])->toArray())->assertOk();

        $integrationInstance = WooCommerceIntegrationInstance::find($response->json()['data']['id']);

        $this->assertEquals((new WooCommerceClient($integrationInstance))
            ->getRedirectUrl(), $response->json('redirect_url'));

        $this->assertDatabaseHas((new IntegrationInstance())->getTable(), [
            'name' => 'WooCommerce',
        ]);

        $this->assertDatabaseHas((new SalesChannel())->getTable(), [
            'integration_instance_id' => $integrationInstance->id,
            'store_id' => Store::first()->id,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Update integration instance
        |--------------------------------------------------------------------------
        */

        $newStore = Store::factory()->create();

        $newIntegrationSettings = [
            'url' => 'https://www.store.com',
            'start_date' => Carbon::now()->subYear()->toDateTimeString(),
            'store_id' => $newStore->id,
            'auto_link_products' => true,
            'auto_create_products' => true,
            'sales_nominal_code_id' => null,
            'cogs_nominal_code_id' => null,
            'pricing' => SalesChannelPricingSettingsData::from([
                'masterOfPrice' => SalesChannelMasterOfPriceSettingsData::from([
                    'name' => 'sku.io',
                ]),
            ]),
            'inventory' => SalesChannelInventorySettingsData::from([
                'masterOfStock' => 'sku.io',
            ]),
            'emailCustomers' => false,
            'proforma_marketplace_cost_percentage' => null,
            'proforma_payment_cost_percentage' => null,
            'sync_sales_order_invoices_to_accounting' => true
        ];

        $this->putJson(route('woo-commerce.update', $integrationInstance->id), UpdateWooCommerceIntegrationData::from([
            'integration_settings' => WooCommerceIntegrationSettingsData::from($newIntegrationSettings),
        ])->toArray())->assertOk();

        $this->assertDatabaseHas((new IntegrationInstance())->getTable(), [
            'name' => 'WooCommerce',
            'integration_settings' => json_encode($newIntegrationSettings),
        ]);

        $this->assertDatabaseHas((new SalesChannel())->getTable(), [
            'integration_instance_id' => $integrationInstance->id,
            'store_id' => $newStore->id,
        ]);

        Queue::assertPushed(CacheProductListingPriceJob::class);
        Queue::assertPushed(GenerateCacheProductListingQuantityJob::class);

        /*
        |--------------------------------------------------------------------------
        | Get redirect Url
        |--------------------------------------------------------------------------
        */

        $this->get(route('woo-commerce.get-redirect-url', $integrationInstance->id))
            ->assertOk()
            ->assertJson([
                'data' => (new WooCommerceClient($integrationInstance))->getRedirectUrl(),
            ]);

        /*
        |--------------------------------------------------------------------------
        | Handle authorization response
        |--------------------------------------------------------------------------
        */

        $callbackData = WooCommerceAuthorizationResponseData::from([
            'key_id' => $this->faker->numberBetween(1, 5000),
            'user_id' => 'subdomain_' . $integrationInstance->id,
            'consumer_key' => 'consumer key',
            'consumer_secret' => 'consumer secret',
            'key_permissions' => 'read_write',
        ]);

        $this->postJson(route('woo-commerce.callback'), $callbackData->toArray())->assertOk();

        $this->assertDatabaseHas((new IntegrationInstance())->getTable(), [
            'id' => $integrationInstance->id,
            'connection_settings' => WooCommerceConnectionSettingsData::from(
                array_merge(['url' => 'https://www.store.com'], $callbackData->toArray())
            )->toJson(),
        ]);
        // Dispatches jobs to get initial products and orders
        Queue::assertPushed(RefreshWooCommerceProductsJob::class);
        Queue::assertPushed(RefreshWooCommerceOrdersJob::class);

        /*
        |--------------------------------------------------------------------------
        | Show integration instance
        |--------------------------------------------------------------------------
        */

        $this->get(route('woo-commerce.show', $integrationInstance->id))
            ->assertOk()
            ->assertJsonStructure([
                'data' => [
                    'id',
                    'name',
                    'connection_settings',
                    'integration_settings',
                    'connection_status',
                    'integration_name',
                    'store',
                    'created_at',
                    'updated_at',
                ],
            ]);

        /*
        |--------------------------------------------------------------------------
        | Delete integration instance
        |--------------------------------------------------------------------------
        */

        $wooCommerceProduct = WooCommerceProduct::factory()->create();
        ProductListing::factory()->create([
            'document_id' => $wooCommerceProduct->id,
            'document_type' => WooCommerceProduct::class,
        ]);
        $wooCommerceOrder = WooCommerceOrder::factory(2)->has(WooCommerceOrderItem::factory(), 'orderItems')->create();
        SalesOrder::factory()->create([
            'sales_channel_order_id' => $wooCommerceOrder->first()->id,
            'sales_channel_order_type' => WooCommerceOrder::class,
        ]);

        $this->assertEquals(1, WooCommerceIntegrationInstance::count());
        $this->assertEquals(1, WooCommerceProduct::count());
        $this->assertEquals(1, ProductListing::count());
        $this->assertEquals(2, WooCommerceOrder::count());
        $this->assertEquals(1, SalesOrder::count());

        $this->delete(route('woo-commerce.destroy', $integrationInstance->id));

        $this->assertEquals(0, WooCommerceIntegrationInstance::query()->count());
        $this->assertEquals(0, WooCommerceProduct::query()->count());
        $this->assertEquals(0, ProductListing::query()->count());
        $this->assertEquals(0, WooCommerceOrder::query()->count());
        $this->assertEquals(0, SalesOrder::query()->count());
    }
}
