<?php

namespace Modules\WooCommerce\Services;

use App\Abstractions\Integrations\AbstractIntegrationClient;
use App\Abstractions\Integrations\Concerns\SimpleAuthenticationTrait;
use App\Abstractions\Integrations\IntegrationClientInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Enums\HttpMethodEnum;
use App\Models\ApiLog;
use Exception;
use Illuminate\Http\Client\PendingRequest;
use Illuminate\Http\Client\RequestException;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Http;
use Modules\WooCommerce\ApiDataTransferObjects\WooCommerceAuthorizationAdt;
use Modules\WooCommerce\Exceptions\WooCommerceReauthorizeException;

/**
 * NOTE: for any WooCommerce installations its possible to override the default number of items returned per page.
 *       For this reason we cant really rely on this default and instead `per_page` is being included in each request URL
 */
class WooCommerceAuthenticationClient extends AbstractIntegrationClient implements IntegrationClientInterface
{
    use SimpleAuthenticationTrait;

    /**
     * @throws Exception
     */
    public function __construct(IntegrationInstanceInterface $integrationInstance)
    {
        parent::__construct($integrationInstance);

        if (
            @$this->integrationInstance->connection_settings['consumer_key']
            && @$this->integrationInstance->connection_settings['consumer_secret']
        ) {
            $this->setClientCredentials(
                $this->integrationInstance->connection_settings['consumer_key'],
                $this->integrationInstance->connection_settings['consumer_secret'],
            );
        }

        $this->setBaseUrl($integrationInstance->integration_settings['url'].config('woo_commerce.endpoint_path'));
    }

    public function client(): PendingRequest
    {
        return Http::withoutVerifying()
            ->throw()
            ->baseUrl($this->getBaseUrl())
            ->withBasicAuth(
                $this->client_id,
                $this->client_secret
            );
    }

    public function getRedirectUrl(): string
    {
        $params = (new WooCommerceAuthorizationAdt($this->integrationInstance))->transform();

        return $this->integrationInstance->integration_settings['url'].config('woo_commerce.authorization_path').'?'.http_build_query($params);
    }

    /**
     * @throws WooCommerceReauthorizeException
     */
    public function request(
        HttpMethodEnum $httpMethod,
        string $path,
        array $parameters = []
        // For GET, this value will be the URL params, E.G ?name=john , For POST this will be JSON in the request body
    ): Response {
        if (is_array($parameters) && count($parameters) > 0) {
            $parameters = collect($parameters)->sortKeys()->toArray();
        }
        $apiLog = null;
        if ($this->isApiLoggingEnabled) {
            $apiLog = ApiLog::query()->create([
                'integration_instance_id' => $this->integrationInstance->id,
                'url' => $path,
                'requestHeaders' => [], // WooCommerce doesnt use headers
                'requestBody' => $parameters,
                'responseStatusCode' => null,
                'responseHeaders' => null,
                'responseBody' => null,
            ]);
        }

        //array_merge is used here to ensure that the consumer_key and consumer_secret are always included in the request
        try {
            $response = $this->client()->{$httpMethod->value}($path, array_merge($parameters, [
                'consumer_key' => $this->client_id,
                'consumer_secret' => $this->client_secret,
            ]));
        } catch (RequestException $e) {
            if ($e->getCode() == 403) {
                throw new WooCommerceReauthorizeException($this->integrationInstance);
            }
        }

        if ($this->isApiLoggingEnabled) {
            $apiLog->update([
                'responseStatusCode' => $response->status(),
                'responseHeaders' => $response->headers(),
                'responseBody' => $response->body(),
            ]);
        }

        return $response;
    }
}
