<?php

namespace Modules\WooCommerce\Entities;

use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelOrderLine;
use App\Data\SalesOrderLineData;
use App\Data\SalesOrderLineTaxLineData;
use App\Data\SalesOrderLineTaxLinesData;
use App\Exceptions\TaxRateException;
use App\Repositories\WarehouseRepository;
use Carbon\CarbonImmutable;
use Exception;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * @property-read WooCommerceOrder $order
 * @property-read WooCommerceProduct $product
 * @property CarbonImmutable|null $created_at
 * @property CarbonImmutable|null $updated_at
 *
 * @method static dataFeedBulkImport(array $array)
 * @method static where(string $string, string $orderItemId)
 */
class WooCommerceOrderItem extends AbstractSalesChannelOrderLine
{
    use HasFactory;

    public static function getUniqueId(): string
    {
        return 'id';
    }

    public static function getTableUniqueId(): string
    {
        return 'line_item_id';
    }

    public static function getLineItemsKey(): ?string
    {
        return 'line_items';
    }

    public static function getParentRelationId(): string
    {
        return 'woo_commerce_order_id';
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function order(): BelongsTo
    {
        return $this->belongsTo(WooCommerceOrder::class, 'woo_commerce_order_id');
    }

    public function product(): \Awobaz\Compoships\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(WooCommerceProduct::class,
            ['integration_instance_id', 'sku'], ['integration_instance_id', 'sku']
        );
    }

    public function getAmount(): float
    {
        return $this->quantity > 0 ? (@$this->json_object['subtotal'] / $this->quantity) : 0;
    }

    public function getTaxAllocation(): float
    {
        return $this->quantity > 0 ? (@$this->json_object['subtotal_tax'] ?? 0) : 0;
    }

    public function getTaxData(): SalesOrderLineTaxLinesData
    {
        $collection = SalesOrderLineTaxLineData::collection(collect(@$this->json_object['taxes'])->map(function ($tax) {
            $taxLine = collect($this->order->json_object['tax_lines'])->where('rate_id', $tax['id'])->first();
            if (!$taxLine) {
                throw new TaxRateException('WooCommerce Tax Line referring to a non existing rate, needs investigation', 0, null, [
                    'tax' => $tax,
                    'taxLines' => $this->order->json_object['tax_lines'],

                ]);
            }
            return new SalesOrderLineTaxLineData(
                rateName: $taxLine['label'],
                rate: $taxLine['rate_percent'],
            );
        }));

        return SalesOrderLineTaxLinesData::from([
            'taxLines' => $collection->count() > 0 ? $collection : null,
            'sales_order_line_id' => $this->{self::getTableUniqueId()},
        ]);
    }

    /**
     * @throws Exception
     */
    public function getWarehouseId(): ?int
    {
        // TODO: This is a temporary patch due to the product relationship not working, please remove
        /** @var WooCommerceProduct $wooCommerceProduct */
        $wooCommerceProduct = WooCommerceProduct::query()
            ->where('sku', $this->sku)
            ->where('integration_instance_id', $this->order->integrationInstance->id)
            ->first();

        $product = $wooCommerceProduct?->productListing?->product;

        if (! $product) {
            return null;
        }

        return (new WarehouseRepository())->getPriorityWarehouseIdForProduct(
            $product,
            $this->quantity,
        );
    }

    /**
     * @throws Exception
     */
    public function getSalesOrderLineData(): SalesOrderLineData
    {
        return SalesOrderLineData::from([
            'sales_order_number' => $this->order->woo_commerce_id,
            'sales_channel_line_id' => $this->{self::getTableUniqueId()},
            'amount' => $this->getAmount(),
            'tax_allocation' => $this->getTaxAllocation(),
            'discount_allocation' => $this->getDiscountAllocation(),
            'description' => $this->name,
            'product_id' => $this->productListingsProductId,
            'product_listing_id' => $this->productListingsId,
            'is_product' => true,
            'quantity' => $this->quantity,
            'externally_fulfilled_quantity' => $this->order->isExternallyFulfilled() ? $this->quantity : 0,
            'warehouse_id' => $this->getWarehouseId(),
            'nominal_code_id' => $this->getNominalCodeId(),
            'tax_data' => $this->getTaxData(),
        ]);
    }

    public function salesChannelProductClass(): string
    {
        return WooCommerceProduct::class;
    }

    public static function getSalesChannelProductUniqueId(): string
    {
        return 'product_id';
    }
}
