<?php

namespace Modules\WooCommerce\Entities;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelOrder;
use App\Abstractions\Integrations\SalesChannels\SalesChannelOrderManagerInterface;
use App\Abstractions\Integrations\SalesChannels\SalesChannelOrderRepositoryInterface;
use App\Data\AddressData;
use App\Data\CustomerData;
use App\Data\FinancialLineData;
use App\DTO\SalesOrderDto;
use App\DTO\SalesOrderStatusesDto;
use App\Enums\FinancialLineClassificationEnum;
use App\Enums\FinancialLineProrationStrategyEnum;
use App\Models\Concerns\BulkImport;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Repositories\FinancialLineRepository;
use Carbon\Carbon;
use Carbon\CarbonImmutable;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Modules\WooCommerce\ApiDataTransferObjects\WooCommerceGetOrdersAdt;
use Modules\WooCommerce\Database\Factories\WooCommerceOrderFactory;
use Modules\WooCommerce\Enums\WooCommerceOrderStatusEnum;
use Modules\WooCommerce\Http\Resources\WooCommerceOrderResource;
use Modules\WooCommerce\Managers\WooCommerceOrderManager;
use Modules\WooCommerce\Repositories\WooCommerceOrderRepository;
use Spatie\LaravelData\DataCollection;
use Throwable;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property string $woo_commerce_id
 * @property string $number
 * @property string $status
 * @property string $error_log
 * @property string $currency
 * @property ?Carbon $date_completed_gmt
 * @property ?Carbon $date_modified_gmt
 * @property ?Carbon $date_created_gmt
 * @property string $total
 * @property string $total_tax
 * @property array $json_object
 * @property-read CarbonImmutable|null $created_at
 * @property CarbonImmutable|null $updated_at
 * @property CarbonImmutable|null $archived_at
 * @property-read WooCommerceIntegrationInstance $integrationInstance
 * @property-read EloquentCollection|WooCommerceOrderItem[] $orderItems
 * @property-read SalesOrder $salesOrder
 */
class WooCommerceOrder extends AbstractSalesChannelOrder
{
    use BulkImport;
    use HasFactory;

    protected $table = 'woo_commerce_orders';

    protected $casts = [
        'json_object' => 'array',
        'error_log' => 'string',
        'error' => 'string',
        'date_completed_gmt' => 'datetime',
        'date_modified_gmt' => 'datetime',
        'date_created_gmt' => 'datetime',
    ];

    public static function getIntegrationName(): string
    {
        return 'woo-commerce';
    }

    public static function getUniqueId(): string
    {
        return 'id';
    }

    public static function getTableUniqueId(): string
    {
        return 'woo_commerce_id';
    }

    public static function getLastModified(): string
    {
        return 'date_modified_gmt';
    }

    public static function repository(): SalesChannelOrderRepositoryInterface
    {
        return app(WooCommerceOrderRepository::class);
    }

    public static function getOrderDate(): string
    {
        return 'date_completed_gmt';
    }

    public static function getCurrency(): ?string
    {
        return 'currency';
    }

    public static function getItemClassName(): string
    {
        return WooCommerceOrderItem::class;
    }

    public function orderItems(): HasMany
    {
        return $this->hasMany(WooCommerceOrderItem::class, 'woo_commerce_order_id');
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function wooCommerceIntegrationInstance(): BelongsTo
    {
        return $this->integrationInstance();
    }

    /*
    |--------------------------------------------------------------------------
    | Methods
    |--------------------------------------------------------------------------
    */

    public function delete(): ?bool
    {
        $this->salesOrder?->delete();
        $this->orderItems()->delete();

        return parent::delete();
    }

    public function generalFilterableColumns(): array
    {
        return ['id', $this->getUniqueId()];
    }

    private function getOrderStatus(): string
    {
        if ($this->status == 'completed' && $this->date_completed_gmt && !$this->isExternallyFulfilled()) {
            // Since it is going to be out of sync for fuflillment status
            return SalesOrder::STATUS_OPEN;
        }
        return match ($this->status) {
            'processing' => SalesOrder::STATUS_OPEN,
            'pending', 'on-hold' => SalesOrder::STATUS_RESERVED,
            /*
             * TODO: Completed and refunded need handling.  For completed, with consideration of SKU-4065, the user
             *  would need to externally fulfill unless there is one warehouse, then maybe we can attempt to
             *  externally fulfill.
             */
            'cancelled', 'refunded', 'failed', 'completed' => SalesOrder::STATUS_CLOSED,
            default => SalesOrder::STATUS_DRAFT,
        };
    }

    private function getFulfillmentStatus(): string
    {
        if ($this->status == 'completed') {
            /*
             * For now, importing historical orders will set the fulfillment status to out of sync so that the user
             * can create fulfillments for it.  This is because we don't know what warehouse the order was fulfilled from,
             * the tracking number, date of fulfillment, etc.
             *
             * Woocommerce does have shipping lines: https://woocommerce.github.io/woocommerce-rest-api-docs/#order-shipping-lines-properties
             * So technically we can create the fulfillment automatically only for single shippable warehouse users... but there is not even
             * a date for the shipping lines... so it may not be possible
             *
             * However, WooCommerce does have a date_completed_gmt field which we can use to determine if the order was fulfilled before
             * the inventory start date.  If so, we can mark as externally fulfilled.
             *
             */
            if ($this->isExternallyFulfilled())
            {
                return SalesOrder::FULFILLMENT_STATUS_FULFILLED;
            }
            return SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC;
        }

        return SalesOrder::FULFILLMENT_STATUS_UNFULFILLED;
    }

    public function isExternallyFulfilled(): bool
    {
        return $this->date_completed_gmt && $this->date_completed_gmt < $this->integrationInstance->integration_settings['start_date'];
    }

    private function getPaymentStatus(): string
    {
        return $this->json_object['needs_payment'] ? SalesOrder::PAYMENT_STATUS_UNPAID : SalesOrder::PAYMENT_STATUS_PAID;
    }

    public function getSalesOrderStatusesDto(): SalesOrderStatusesDto
    {
        return SalesOrderStatusesDto::from([
            'order_status' => $this->getOrderStatus(),
            'fulfillment_status' => $this->getFulfillmentStatus(),
            'payment_status' => $this->getPaymentStatus(),
        ]);
    }

    public function getCustomerDto(): ?CustomerData
    {
        $json_object = $this->json_object;

        if (@$json_object['billing']) {
            return CustomerData::from([
                'name' => $json_object['billing']['first_name'].' '.$json_object['billing']['last_name'],
                'email' => $json_object['billing']['email'],
                'zip' => $json_object['billing']['postcode'],
                'address1' => $json_object['billing']['address_1'],
                'address2' => $json_object['billing']['address_2'],
                'city' => $json_object['billing']['city'],
                'province' => $json_object['billing']['state'],
                'country' => $json_object['billing']['country'],
                'company' => $json_object['billing']['company'],
                'fax' => null,
                'sales_channel_origin_id' => $this->integrationInstance->salesChannel->id,
                'default_shipping_address' => AddressData::from([
                    'name' => $json_object['shipping']['first_name'].' '.$json_object['shipping']['last_name'],
                    'email' => $json_object['billing']['email'],
                    'zip' => $json_object['shipping']['postcode'],
                    'label' => 'Default Address',
                    'address1' => $json_object['shipping']['address_1'],
                    'address2' => $json_object['shipping']['address_2'],
                    'city' => $json_object['shipping']['city'],
                    'province' => $json_object['shipping']['state'],
                    'province_code' => $json_object['shipping']['state'],
                    'country' => $json_object['shipping']['country'],
                    'country_code' => $json_object['shipping']['country'],
                ]),
                'default_billing_address' => AddressData::from([
                    'name' => $json_object['billing']['first_name'].' '.$json_object['billing']['last_name'],
                    'email' => $json_object['billing']['email'],
                    'zip' => $json_object['billing']['postcode'],
                    'label' => 'Default Address',
                    'address1' => $json_object['billing']['address_1'],
                    'address2' => $json_object['billing']['address_2'],
                    'city' => $json_object['billing']['city'],
                    'province' => $json_object['billing']['state'],
                    'province_code' => $json_object['billing']['state'],
                    'country' => $json_object['billing']['country'],
                    'country_code' => $json_object['billing']['country'],
                ]),
            ]);
        }

        return null;
    }

    private function getRequestedShippingMethod(): string
    {
        return @$this->json_object['shipping_lines'][0]['method_title'] ?? SalesChannel::UNSPECIFIED_SHIPPING_METHOD;
    }

    private function getCanceledAt(): ?Carbon
    {
        return $this->status === 'cancelled' || $this->status === 'failed' ? Carbon::parse($this->date_modified_gmt) : null;
    }

    /**
     * @throws Throwable
     */
    public function getSalesOrderDto(): SalesOrderDto
    {
        $orderStatuses = $this->getSalesOrderStatusesDto();

        return SalesOrderDto::from([
            'sales_order_number' => $this->number,
            'sales_channel_id' => $this->integrationInstance->salesChannel->id,
            'currency_code' => $this->getCurrencyCode($this->currency),
            'order_date' => Carbon::parse($this->date_created_gmt),
            // TODO: Could support mapping between $record->shipping_lines[*]->method_id and shipping_method_id
            'shipping_method_id' => null,
            'requested_shipping_method' => $this->getRequestedShippingMethod(),
            'order_status' => $orderStatuses->order_status,
            'fulfillment_status' => $orderStatuses->fulfillment_status,
            'payment_status' => $orderStatuses->payment_status,
            'canceled_at' => $this->getCanceledAt(),
            'sales_order_lines' => $this->getSalesOrderLines(),
            'financial_lines' => $this->getFinancialLines(),
            'sales_channel_order_type' => self::class,
            'sales_channel_order_id' => $this->id,
            'customer' => $this->getCustomerDto(),
            'last_synced_from_sales_channel_at' => $this->updated_at,
        ]);
    }

    public function getFinancialLines(): DataCollection
    {
        $financialLines = parent::getFinancialLines();

        // TODO: Add a fixed fee setting as well (i.e. 0.30)
        // TODO: Integrate directly with paypal?
        if ($marketplaceCostPercentage = @$this->integrationInstance->integration_settings['proforma_payment_cost_percentage']) {
            $financialLines[] = FinancialLineData::from([
                'sales_order_id' => $this->id,
                'financial_line_type_id' => app(FinancialLineRepository::class)
                    ->getOrCreateFinancialLineType(
                        'Payment Fee',
                        FinancialLineClassificationEnum::COST,
                        prorationStrategy: FinancialLineProrationStrategyEnum::REVENUE_BASED,
                        allocateToProducts: true
                    )->id,
                'description' => 'WooCommerce Payment Processing Fee',
                'quantity' => 1,
                'amount' => $this->total * ($marketplaceCostPercentage / 100),
                'tax_allocation' => 0,
                'allocate_to_products' => 1,
                'proration_strategy' => FinancialLineProrationStrategyEnum::REVENUE_BASED,
            ]);
        }

        return $financialLines;
    }

    public function getShippingTotal(): float
    {
        return $this->shipping_total ?? 0;
    }

    public function getShippingTaxTotal(): float
    {
        return $this->shipping_tax;
    }

    public function getDiscountTotal(): float
    {
        return -$this->discount_total;
    }

    public function getDiscountTaxTotal(): float
    {
        return -$this->discount_tax;
    }

    public static function newFactory(): Factory
    {
        return WooCommerceOrderFactory::new();
    }

    public static function getResource()
    {
        return WooCommerceOrderResource::class;
    }

    public function isFullyShipped(): bool
    {
        return $this->status === WooCommerceOrderStatusEnum::COMPLETED();
    }

    public function isPartiallyShipped(): bool
    {
        return false;
    }

    public function partiallyFulfill(SalesOrder $salesOrder)
    {
    }

    //TODO:Need to work on this
    public function getSkuOrderLines(): array
    {
        $salesChannel = $this->integrationInstance->salesChannel;
        $lines = [];

        foreach ($this->orderItems as $item) {
            $listing = ProductListing::with([])->where('listing_sku', $item['sku'])
                ->where('sales_channel_listing_id', $item['sku'])
                ->where('sales_channel_id', $salesChannel->id)
                ->first();

            $line = null;
        }

        return $lines;
    }

    public function getSkuShippingMethodId()
    {
        return $this->getSkuShippingMethod();
    }

    // TODO: Add contract for the following methods to the interface once implemented everywhere

    public function manager(WooCommerceIntegrationInstance|IntegrationInstanceInterface $integrationInstance): SalesChannelOrderManagerInterface
    {
        return new WooCommerceOrderManager($integrationInstance);
    }

    public function refreshAdt(): ApiDataTransformerInterface
    {
        return new WooCommerceGetOrdersAdt();
    }
}
