<?php

namespace Modules\WooCommerce\Console;

use App\Abstractions\Integrations\Data\SalesChannelIntegrationSettingsData;
use App\Data\SalesChannelInventorySettingsData;
use App\Data\SalesChannelMasterOfPriceSettingsData;
use App\Data\SalesChannelPricingSettingsData;
use App\Models\IntegrationInstance;
use App\Repositories\SalesChannelRepository;
use Carbon\Carbon;
use Exception;
use Illuminate\Console\Command;
use Modules\Ebay\Data\StoreEbayIntegrationData;
use Modules\WooCommerce\Data\StoreWooCommerceIntegrationData;
use Modules\WooCommerce\Data\WooCommerceConnectionSettingsData;
use Modules\WooCommerce\Entities\WooCommerceIntegration;
use Modules\WooCommerce\Repositories\WooCommerceIntegrationInstanceRepository;
use Modules\WooCommerce\Services\WooCommerceClient;

class InitializeWoocommerceIntegrationInstanceCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'woo-commerce:initialize
                                {url : The URL of the WooCommerce store (www.site.com)}
                                {name=WooCommerce : The name of the integration instance}
                                {store_id=1 : The ID of the store}';

    protected $description = 'Create Woocommerce integration instance and provide authorization URL';

    protected WooCommerceIntegrationInstanceRepository $integrationInstances;

    protected SalesChannelRepository $salesChannels;

    public function __construct()
    {
        parent::__construct();

        $this->integrationInstances = app(WooCommerceIntegrationInstanceRepository::class);
        $this->salesChannels        = app(SalesChannelRepository::class);
    }

    /**
     * @throws Exception
     */
    public function handle(): void
    {
        $name = $this->argument('name');
        $storeId = $this->argument('store_id');

        $wooCommerceIntegrationInstanceData = StoreWooCommerceIntegrationData::from([
            'integration_id' => WooCommerceIntegration::sole()->id,
            'name' => $name,
            'connection_settings' => WooCommerceConnectionSettingsData::from([
                'url' => $this->argument('url'),
            ]),
            'integration_settings' => SalesChannelIntegrationSettingsData::from([
                'start_date' => Carbon::now()->subYear()->startOfYear(),
                'store_id' => $storeId,
                'auto_link_products' => false,
                'auto_create_products' => false,
                'sales_nominal_code_id' => null,
                'cogs_nominal_code_id' => null,
                'pricing' => SalesChannelPricingSettingsData::from([
                    'masterOfPrice' => SalesChannelMasterOfPriceSettingsData::from([])
                ]),
                'inventory' => SalesChannelInventorySettingsData::from([]),
                'emailCustomers' => false,
                'proforma_marketplace_cost_percentage' => null,
                'proforma_payment_cost_percentage' => null,
            ]),
        ]);

        $wooCommerceIntegrationInstance = IntegrationInstance::create($wooCommerceIntegrationInstanceData->toArray());

        $this->salesChannels->saveFromIntegrationInstance($wooCommerceIntegrationInstance);

        $wooCommerceClient = new WooCommerceClient($wooCommerceIntegrationInstance);
        $redirectUrl = $wooCommerceClient->getRedirectUrl();

        $this->line("Visit the following URL to complete the setup:\n");
        $this->line($redirectUrl);
    }
}
