<?php

namespace Modules\Veracore\Tests\Unit;

use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Http;
use Modules\Veracore\Entities\VeracoreIntegrationInstance;
use Modules\Veracore\Services\VeracoreClient;
use Modules\Veracore\Services\VeracoreManager;
use Modules\Veracore\Tests\VeracoreTest;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;

class VeracoreUpdateOrderTest extends VeracoreTest
{

    use FastRefreshDatabase;
    use WithFaker;


    private SalesOrder $order;
    private int $veracoreOrderId = 1;
    private FulfillmentManager $fulfillmentManager;
    private SalesOrderFulfillment $fulfillment;

    protected function setUp(): void
    {
        parent::setUp();

        $this->order = SalesOrder::factory()->reserved(2);
        $this->order->update(['deliver_by_date' => now()->addDays(5)]);
        /** @var SalesOrderLine $firstOrderLine */
        $firstOrderLine = $this->order->salesOrderLines->first();

        $veracore = VeracoreIntegrationInstance::active();
        $settings = $veracore->integration_settings;
        $connectionSettings = $veracore->connection_settings;
        $connectionSettings['access_token'] = $this->faker->md5();
        $settings['linked_warehouse_id'] = $firstOrderLine->warehouse_id;
        $veracore->update(['integration_settings' => $settings, 'connection_settings' => $connectionSettings]);

        Http::fake([
            VeracoreClient::makeUrl(VeracoreClient::BASE_ENDPOINT) => Http::response(
                '<?xml version="1.0" encoding="utf-8"?>
                        <soap:Envelope xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema">
                            <soap:Body>
                                <AddOrderResponse xmlns="http://omscom/">
                                    <AddOrderResult>
                                        <OrderSeqID>183</OrderSeqID>
                                        <OrderID>'.$this->veracoreOrderId.'</OrderID>
                                    </AddOrderResult>
                                </AddOrderResponse>
                            </soap:Body>
                        </soap:Envelope>'
            ),
            VeracoreClient::makeUrl(VeracoreClient::BASE_ENDPOINT.'?op=GetOrderInfo') => Http::response(
                '<?xml version="1.0" encoding="utf-8"?>
                        <soap:Envelope xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema">
                            <soap:Header>
                                <DebugHeader xmlns="http://omscom/">
                                    <Debug>false</Debug>
                                </DebugHeader>
                            </soap:Header>
                            <soap:Body>
                                <GetOrderInfoResponse xmlns="http://omscom/">
                                    <GetOrderInfoResult>
                                        <OrdHead>
                                            <OrderId>'.$this->veracoreOrderId.'</OrderId>
                                            <Status>
                                                <Unapproved>false</Unapproved>
                                                <Unprocessed>true</Unprocessed>
                                            </Status>
                                        </OrdHead>
                                        <OfferInfo>
                                            <OfferType>
                                                <OfferId>'.$this->order->salesOrderLines->first()->product->sku.'</OfferId>
                                                <OfferDesc>'.$this->order->salesOrderLines->first()->description.'</OfferDesc>
                                                <OrderQty>1</OrderQty>
                                                <UnitPrice>'.$this->order->salesOrderLines->first()->amount.'</UnitPrice>
                                            </OfferType>
                                        </OfferInfo>
                                    </GetOrderInfoResult>
                                </GetOrderInfoResponse>
                            </soap:Body>
                        </soap:Envelope>
                    '
            ),
            VeracoreClient::makeUrl(VeracoreClient::GET_PACKAGES_ENDPOINT) . '?*' => Http::response([
                'ShippingUnits' => [
                    [
                        'Order' => [
                            'OrderID' => 121,
                            'ReferenceNumber' => VeracoreManager::VERACORE_TEST_REFERENCE,
                            'CurrentOrderStatus' => 'Fulfilled',
                        ],
                        'Shipping' => [
                            'UTCShippedDateTime' => now()->subDay(),
                            'FreightCarrier' => 'UPS',
                            'FreightService' => 'STANDARD',
                            'FreightCodeDescription' => 'Ground',
                            'Freight' => [
                                'MarkedUpFreight' => $this->faker->numberBetween(3, 8)
                            ],
                            'TrackingNumber' => $this->faker->md5(),
                            'ShipTo' => [
                                'Company' => "Test\nCompany",
                            ]
                        ]
                    ],
                ]
            ])
        ]);

        $this->fulfillmentManager = app(FulfillmentManager::class);
        $this->fulfillment = $this->fulfillmentManager->fulfill($this->order, [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_VERACORE,
            'warehouse_id' => $this->order->salesOrderLines->first()->warehouse_id,
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $firstOrderLine->id,
                    'quantity' => 1,
                ],
            ],
        ]);
        $this->assertDatabaseCount('veracore_orders', 1);
        $this->assertDatabaseHas('veracore_orders', [
            'sku_fulfillment_id' => $this->fulfillment->id,
            'veracore_id' => $this->veracoreOrderId,
        ]);


    }


    public function test_it_adds_new_line_to_veracore_fulfillment(): void
    {
        $this->fulfillmentManager->updateFulfillment($this->fulfillment, [
            'fulfillment_lines' => [
                [
                    'id' => $this->fulfillment->salesOrderFulfillmentLines->first()->id,
                    'quantity' => 1,
                ],
                [
                    'sales_order_line_id' => $this->order->salesOrderLines->last()->id,
                    'quantity' => 1,
                ],
            ],
        ]);

        // Mock the get order detail endpoint
        Http::fake(
            [
                VeracoreClient::makeUrl(VeracoreClient::ORDER_ADJUSTMENTS_ENDPOINT) => Http::response(
                    '<s:Envelope xmlns:s="http://schemas.xmlsoap.org/soap/envelope/">
                           <s:Body>
                              <AddOfferResponse xmlns="http://omscom/">
                                 <AddOfferResult xmlns:a="http://schemas.datacontract.org/2004/07/ProMail.WebServices.Objects.OrderAdjustments" xmlns:i="http://www.w3.org/2001/XMLSchema-instance">
                                    <a:ExceptionMessage i:nil="true"/>
                                    <a:OrderID>'.$this->order->salesOrderLines->last()->product->sku.'</a:OrderID>
                                    <a:OrderSeqID>123</a:OrderSeqID>
                                 </AddOfferResult>
                              </AddOfferResponse>
                           </s:Body>
                        </s:Envelope>
                    '
                )
            ]
        );

        // Update the veracore order
        $this->putJson("/api/veracore/update-order/{$this->fulfillment->id}")->assertSuccessful();
        Http::assertSentCount(2);
    }

    public function test_it_updates_veracore_order_quantity_when_sku_fulfillment_quantity_increases(): void{

        $this->fulfillmentManager->updateFulfillment($this->fulfillment, [
            'fulfillment_lines' => [
                [
                    'id' => $this->fulfillment->salesOrderFulfillmentLines->first()->id,
                    'quantity' => 2, // Increase quantity
                ]
            ],
        ]);

        // Mock the increase order quantity endpoint
        Http::fake(
            [
                VeracoreClient::makeUrl(VeracoreClient::ORDER_ADJUSTMENTS_ENDPOINT) => Http::response(
                    '<s:Envelope xmlns:s="http://schemas.xmlsoap.org/soap/envelope/">
                           <s:Body>
                              <UpdateOfferQuantityResponse xmlns="http://omscom/">
                                 <UpdateOfferQuantityResult xmlns:a="http://schemas.datacontract.org/2004/07/ProMail.WebServices.Objects.OrderAdjustments" xmlns:i="http://www.w3.org/2001/XMLSchema-instance">
                                    <a:ExceptionMessage i:nil="true"/>
                                    <a:OrderID>'.$this->order->salesOrderLines->first()->product->sku.'</a:OrderID>
                                    <a:OrderSeqID>123</a:OrderSeqID>
                                 </UpdateOfferQuantityResult>
                              </UpdateOfferQuantityResponse>
                           </s:Body>
                        </s:Envelope>
                    '
                )
            ]
        );

        // Update the veracore order
        $this->putJson("/api/veracore/update-order/{$this->fulfillment->id}")
            ->assertSuccessful();

        Http::assertSentCount(2);

    }

    public function test_it_updates_veracore_order_quantity_when_sku_fulfillment_quantity_decreases(): void{

        $this->fulfillmentManager->updateFulfillment($this->fulfillment, [
            'fulfillment_lines' => [
                [
                    'id' => $this->fulfillment->salesOrderFulfillmentLines->first()->id,
                    'quantity' => 0, // Decrease quantity
                ]
            ],
        ]);

        // Mock the decrease order quantity endpoint
        Http::fake(
            [
                VeracoreClient::makeUrl(VeracoreClient::ORDER_ADJUSTMENTS_ENDPOINT) => Http::response(
                    '<s:Envelope xmlns:s="http://schemas.xmlsoap.org/soap/envelope/">
                           <s:Body>
                              <UpdateOfferQuantityResponse xmlns="http://omscom/">
                                 <UpdateOfferQuantityResult xmlns:a="http://schemas.datacontract.org/2004/07/ProMail.WebServices.Objects.OrderAdjustments" xmlns:i="http://www.w3.org/2001/XMLSchema-instance">
                                    <a:ExceptionMessage i:nil="true"/>
                                    <a:OrderID>'.$this->order->salesOrderLines->first()->product->sku.'</a:OrderID>
                                    <a:OrderSeqID>123</a:OrderSeqID>
                                 </UpdateOfferQuantityResult>
                              </UpdateOfferQuantityResponse>
                           </s:Body>
                        </s:Envelope>
                    '
                )
            ]
        );

        // Update the veracore order
        $this->putJson("/api/veracore/update-order/{$this->fulfillment->id}")
            ->assertSuccessful();

        Http::assertSentCount(2);

    }

}
