<?php

namespace Modules\Veracore\Tests\Feature;

use App\Models\CustomField;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\User;
use App\Models\Warehouse;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Testing\WithFaker;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class VeracoreIntegrationInstanceControllerTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    protected Warehouse $warehouse;
    protected Integration|Model $integration;

    protected function setUp(): void
    {
        parent::setUp();
        $this->actingAs(User::factory()->create());
        $this->warehouse = Warehouse::factory()->create();
        $this->integration = Integration::query()->firstOrCreate([
            'name' => Integration::NAME_VERACORE
        ]);
    }

    public function test_it_can_create_veracore_integration_instance(): void{

        $this->postJson(route('veracore.integration-instances.store'), [
            'integration_id' => Integration::factory()->create()->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true,
                'is_deliver_by_date_required' => true,
                'gift_card_note_sales_order_custom_field_id' => null
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'systemId' => $this->faker->uuid(),
            ]
        ])->assertSuccessful()->assertJsonStructure([
            'data' => [
                'id',
                'integration_id',
                'name',
                'integration_settings' => [
                    'connection_name',
                    'linked_warehouse',
                    'automate_fulfillment',
                    'is_deliver_by_date_required',
                    'gift_card_note_sales_order_custom_field_id'
                ],
                'connection_settings' => [
                    'username',
                    'password',
                    'systemId'
                ],
                'created_at',
                'updated_at'
            ]
        ]);

        $this->assertDatabaseHas('integration_instances', [
            'name' => 'Veracore',
        ]);

    }

    public function test_it_can_get_veracore_integration_instance(): void{

        $response = $this->postJson(route('veracore.integration-instances.store'), [
            'integration_id' => Integration::factory()->create()->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'systemId' => $this->faker->uuid(),
            ]
        ])->assertSuccessful();

        $this->getJson(route('veracore.integration-instances.show', $response->json('data.id')))
            ->assertSuccessful()->assertJsonStructure([
                'data' => [
                    'id',
                    'integration_id',
                    'name',
                    'integration_settings' => [
                        'connection_name',
                        'linked_warehouse',
                        'automate_fulfillment',
                        'is_deliver_by_date_required',
                        'gift_card_note_sales_order_custom_field_id'
                    ],
                    'connection_settings',
                    'created_at',
                    'updated_at'
                ]
            ]);
    }

    public function test_it_cannot_create_veracore_instance_without_linked_warehouse_id(): void{

        $this->postJson(route('veracore.integration-instances.store'), [
            'integration_id' => Integration::factory()->create()->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'systemId' => $this->faker->uuid(),
            ]
        ])->assertStatus(422);
    }

    public function test_it_cannot_create_multiple_veracore_instances(): void{

        $this->postJson(route('veracore.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'systemId' => $this->faker->uuid(),
            ]
        ])->assertSuccessful();

        $this->postJson(route('veracore.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'systemId' => $this->faker->uuid(),
            ]
        ])->assertStatus(422);
    }


    public function test_it_can_update_veracore_integration_instance(): void{

        $response = $this->postJson(route('veracore.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'systemId' => $this->faker->uuid(),
            ]
        ])->assertSuccessful();

        $this->putJson(route('veracore.integration-instances.update', $response->json('data.id')), [
            'integration_id' => $this->integration->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'systemId' => $this->faker->uuid(),
            ]
        ])->assertSuccessful()->assertJsonStructure([
            'data' => [
                'id',
                'integration_id',
                'name',
                'integration_settings' => [
                    'connection_name',
                    'linked_warehouse',
                    'automate_fulfillment',
                    'is_deliver_by_date_required',
                    'gift_card_note_sales_order_custom_field_id'
                ],
                'connection_settings',
                'created_at',
                'updated_at'
            ]
        ]);

        $this->assertDatabaseHas('integration_instances', [
            'name' => 'Veracore',
        ]);

    }

    public function test_it_can_update_gift_card_note_sales_order_custom_field_id(): void{

        $response = $this->postJson(route('veracore.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true,
                'is_deliver_by_date_required' => true,
                'gift_card_note_sales_order_custom_field_id' => null
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'systemId' => $this->faker->uuid(),
            ]
        ])->assertSuccessful();

        $this->putJson(route('veracore.integration-instances.update', $response->json('data.id')), [
            'integration_id' => $this->integration->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true,
                'is_deliver_by_date_required' => true,
                'gift_card_note_sales_order_custom_field_id' => 1
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'systemId' => $this->faker->uuid(),
            ]
        ])->assertSuccessful()->assertJsonStructure([
            'data' => [
                'id',
                'integration_id',
                'name',
                'integration_settings' => [
                    'connection_name',
                    'linked_warehouse',
                    'automate_fulfillment',
                    'is_deliver_by_date_required',
                    'gift_card_note_sales_order_custom_field_id'
                ],
                'connection_settings',
                'created_at',
                'updated_at'
            ]
        ]);

        $this->assertDatabaseHas('integration_instances', [
            'name' => 'Veracore',
            'integration_settings->gift_card_note_sales_order_custom_field_id' => 1
        ]);

    }

    public function test_it_can_set_gift_card_note_sales_order_custom_field_id_to_none(): void{

        $response = $this->postJson(route('veracore.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true,
                'is_deliver_by_date_required' => true,
                'gift_card_note_sales_order_custom_field_id' => 1
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'systemId' => $this->faker->uuid(),
            ]
        ])->assertSuccessful();

        $this->putJson(route('veracore.integration-instances.update', $response->json('data.id')), [
            'integration_id' => $this->integration->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true,
                'is_deliver_by_date_required' => true,
                'gift_card_note_sales_order_custom_field_id' => CustomField::FIELD_NONE
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'systemId' => $this->faker->uuid(),
            ]
        ])->assertSuccessful()->assertJsonStructure([
            'data' => [
                'id',
                'integration_id',
                'name',
                'integration_settings' => [
                    'connection_name',
                    'linked_warehouse',
                    'automate_fulfillment',
                    'is_deliver_by_date_required',
                    'gift_card_note_sales_order_custom_field_id'
                ],
                'connection_settings',
                'created_at',
                'updated_at'
            ]
        ]);

        $this->assertDatabaseHas('integration_instances', [
            'name' => 'Veracore',
            'integration_settings->gift_card_note_sales_order_custom_field_id' => CustomField::FIELD_NONE
        ]);

    }

    public function test_it_can_change_name_for_same_veracore_instance(): void{

        $response = $this->postJson(route('veracore.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'systemId' => $this->faker->uuid(),
            ]
        ])->assertSuccessful();

        $this->putJson(route('veracore.integration-instances.update', $response->json('data.id')), [
            'integration_id' => $this->integration->id,
            'name' => 'Veracore2',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'systemId' => $this->faker->uuid(),
            ]
        ])->assertSuccessful()->assertJsonStructure([
            'data' => [
                'id',
                'integration_id',
                'name',
                'integration_settings',
                'connection_settings',
                'created_at',
                'updated_at'
            ]
        ]);

        $this->assertDatabaseHas('integration_instances', [
            'name' => 'Veracore2',
        ]);
    }

    public function test_it_can_delete_veracore_instance(): void{

        $response = $this->postJson(route('veracore.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'systemId' => $this->faker->uuid(),
            ]
        ])->assertSuccessful();

        $this->deleteJson(route('veracore.integration-instances.destroy', $response->json('data.id')))
            ->assertSuccessful();

        $this->assertDatabaseMissing('integration_instances', [
            'name' => 'Veracore',
        ]);
    }

    public function test_it_can_change_linked_direct_warehouse_to_3pl_warehouse(): void{

        $this->postJson(route('veracore.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'systemId' => $this->faker->uuid(),
            ]
        ])->assertSuccessful();


        $this->assertDatabaseHas('warehouses', [
            'id' => $this->warehouse->id,
            'type' => Warehouse::TYPE_3PL
        ]);
    }

    public function test_only_direct_or_3pl_warehouse_can_be_linked(): void{

        $this->postJson(route('veracore.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => Warehouse::factory()->create([
                    'type' => Warehouse::TYPE_SUPPLIER
                ])->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'systemId' => $this->faker->uuid(),
            ]
        ])->assertStatus(422);
    }

    public function test_it_does_not_change_is_automatic_sync_enabled_setting_when_connection_settings_change(): void{

        $response = $this->postJson(route('veracore.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => false
            ],
            'connection_settings' => [
                'username' => 'test',
                'password' => 'test',
                'systemId' => 'test',
            ],
            'is_automatic_sync_enabled' => true
        ])->assertSuccessful();

        $this->putJson(route('veracore.integration-instances.update', $response->json('data.id')), [
            'integration_id' => $this->integration->id,
            'name' => 'Veracore',
            'integration_settings' => [
                'connection_name' => 'Veracore',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => false
            ],
            'connection_settings' => [
                'username' => 'test2',
                'password' => 'test2',
                'systemId' => 'test2',
            ]
        ])->assertSuccessful()->assertJsonStructure([
            'data' => [
                'id',
                'integration_id',
                'name',
                'integration_settings',
                'connection_settings',
                'created_at',
                'updated_at'
            ]
        ]);

        $this->assertDatabaseHas('integration_instances', [
            'name' => 'Veracore',
            'is_automatic_sync_enabled' => true
        ]);

    }

    public function test_it_can_change_connection_settings_only_without_specifying_integration_settings(): void{

            $instance = IntegrationInstance::factory()->create([
                'integration_id' => $this->integration->id,
                'name' => 'Veracore',
                'integration_settings' => [
                    'connection_name' => 'Veracore',
                    'linked_warehouse_id' => $this->warehouse->id,
                    'automate_fulfillment' => false
                ],
                'connection_settings' => [
                    'username' => 'test',
                    'password' => 'test',
                    'systemId' => 'test',
                ],
                'is_automatic_sync_enabled' => true
            ]);

            $this->putJson(route('veracore.integration-instances.update', $instance->id), [
                'integration_id' => $this->integration->id,
                'name' => 'Veracore',
                'connection_settings' => [
                    'username' => 'test2',
                    'password' => 'test2',
                    'systemId' => 'test2',
                ]
            ])->assertSuccessful()->assertJsonStructure([
                'data' => [
                    'id',
                    'integration_id',
                    'name',
                    'integration_settings',
                    'connection_settings',
                    'created_at',
                    'updated_at'
                ]
            ]);

            $this->assertDatabaseHas('integration_instances', [
                'name' => 'Veracore',
                'connection_settings->username' => 'test2'
            ]);
    }

    public function test_it_can_turn_off_is_deliver_by_date_required_setting(): void{

            $response = $this->postJson(route('veracore.integration-instances.store'), [
                'integration_id' => $this->integration->id,
                'name' => 'Veracore',
                'integration_settings' => [
                    'connection_name' => 'Veracore',
                    'linked_warehouse_id' => $this->warehouse->id,
                    'automate_fulfillment' => false,
                    'is_deliver_by_date_required' => true,
                    'gift_card_note_sales_order_custom_field_id' => null
                ],
                'connection_settings' => [
                    'username' => 'test',
                    'password' => 'test',
                    'systemId' => 'test',
                ],
                'is_automatic_sync_enabled' => true
            ])->assertSuccessful();

            $this->putJson(route('veracore.integration-instances.update', $response->json('data.id')), [
                'integration_id' => $this->integration->id,
                'name' => 'Veracore',
                'integration_settings' => [
                    'connection_name' => 'Veracore',
                    'linked_warehouse_id' => $this->warehouse->id,
                    'automate_fulfillment' => false,
                    'is_deliver_by_date_required' => false,
                    'gift_card_note_sales_order_custom_field_id' => null
                ],
                'connection_settings' => [
                    'username' => 'test2',
                    'password' => 'test2',
                    'systemId' => 'test2',
                ]
            ])->assertSuccessful()->assertJsonStructure([
                'data' => [
                    'id',
                    'integration_id',
                    'name',
                    'integration_settings',
                    'connection_settings',
                    'created_at',
                    'updated_at'
                ]
            ]);

            $this->assertFalse(
                IntegrationInstance::query()
                    ->where('id', $response->json('data.id'))
                    ->firstOrFail()->integration_settings['is_deliver_by_date_required']
            );
    }
}
