<?php

namespace Modules\Veracore\Services;

use App\Helpers\ValidatesXml;
use App\Integrations\Http\IntegrationHttpClient;
use App\Integrations\Http\IntegrationTimeoutException;
use App\Jobs\UpdateApiLogJob;
use App\Models\ApiLog;
use App\Services\ShippingProvider\ShippingProviderClient;
use Exception;
use Illuminate\Http\Client\Pool;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\View;
use Modules\Veracore\Exceptions\VeracoreCannotFetchOrderException;
use RuntimeException;
use Throwable;

class VeracoreClient extends IntegrationHttpClient implements ShippingProviderClient
{
    use ValidatesXml;

    const BASE_ENDPOINT = '/pmomsws/oms.asmx';

    const GET_PACKAGES_ENDPOINT = '/VeraCore/Public.Api/api/GetPackages';

    const ACCESS_TOKEN_ENDPOINT = '/VeraCore/Public.Api/api/Login';

    const ORDER_ADJUSTMENTS_ENDPOINT = '/pmomsws/orderadjustment.svc/soap';

    public function __construct(
        private readonly CredentialManager $credential,
    ) {
        parent::__construct($this->credential->getIntegrationInstance());
    }

    /**
     * @throws Throwable
     */
    public function getTrackingInfo(array $needingUpdates): array
    {

        // Split the requests into chunks of 100
        // and use HTTP client pool to make the requests asynchronously
        $chunks = array_chunk($needingUpdates, 100);
        $tracking = [];
        $errors = [];
        $headers = [
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer '.$this->getAccessToken(),
        ];

        $urls = [];
        $apiLogs = [];
        foreach ($chunks as $chunk) {
            $query = ['OrdersIds' => implode(',', $chunk)];
            $url = self::makeUrl(self::GET_PACKAGES_ENDPOINT).'?'.$this->buildQuery($query);
            $urls[] = $url;
            $apiLogs[] = $this->logger->logApiCall(
                integrationInstance: $this->credential->getIntegrationInstance(),
                url: explode('?', $url)[0],
                requestBody: json_encode($query),
                requestHeaders: $headers
            );
        }

        $responses = Http::pool(function (Pool $pool) use ($urls, $headers) {
            return collect($urls)->map(
                fn ($url) => $pool->withHeaders($headers)->get($url)
            )->all();
        });

        foreach ($responses as $index => $response) {

            /** @var ApiLog $log */
            $log = $apiLogs[$index];

            if ($response instanceof Exception) {
                $log->update([
                    'responseStatusCode' => 500,
                    'responseBody' => $response->getMessage(),
                ]);
                $errors[] = $response->getMessage();

                continue;
            }

            $log->update([
                'responseStatusCode' => $response->status(),
                'responseHeaders' => $response->headers(),
                'responseBody' => $response->body(),
            ]);

            if ($response->successful()) {
                if (! empty($response->json()['ShippingUnits'])) {
                    $results = $response->json()['ShippingUnits'];
                    array_push($tracking, ...$results);
                }
            } else {
                $errors[] = $response->reason();
            }
        }

        if (count($errors) == count($chunks)) {
            // All the requests failed
            throw new RuntimeException(
                'Error getting tracking info from Veracore: '.implode(', ', $errors)
            );
        }

        return $tracking;
    }

    /**
     * @throws Throwable
     */
    public function getOrder(int $orderId): array
    {

        $headers = [
            'Content-Type' => 'text/xml',
            'Authorization' => 'Bearer '.$this->getAccessToken(),
        ];

        $payload = View::make(
            'veracore::get_order',
            array_merge(
                $this->getAuthenticationPayload(),
                ['id' => $orderId]
            )
        )->render();

        $url = $this->makeUrl(self::BASE_ENDPOINT.'?op=GetOrderInfo');

        $response = $this->post(
            url: $url,
            body: $payload,
            bodyContentType: 'text/xml',
            headers: $headers,
            tries: 3
        );

        if ($response->successful()) {
            // Extract returned ID and return it.
            $desiredXPath = '//*[local-name()="GetOrderInfoResult"]';

            $parsed = $this->parseResponseXPaths(
                $response->body(),
                $desiredXPath,
                [],
                false
            );

            $parsed = json_decode(json_encode($parsed[$desiredXPath][0]), true) ?? [];
            if (! empty($parsed)) {

                if (! isset($parsed['OfferInfo']['OfferType'][0])) {
                    $parsed['OfferInfo']['OfferType'] = [$parsed['OfferInfo']['OfferType']];
                }

            }

            return $parsed;
        }

        throw new RuntimeException(
            'Error getting order from Veracore: '.$response->reason()
        );

    }

    public static function makeUrl($path): string
    {
        return config('veracore.base_url').$path;
    }

    /**
     * @throws Exception|Throwable
     */
    public function sendFulfillmentOrder(array $payload): int
    {

        // Check if the order already exists in Veracore
        // If it does, return the order ID.
        // This prevents duplicate orders from being created in Veracore.
        // We use the order ID and the reference number to check if the order
        // already exists and was originated from us.
        $id = $payload['id'];
        $reference = $payload['reference'];
        if ($order = $this->findExistingOrderInVeracore($id, $reference)) {
            return $order['OrderId'];
        }

        $payload = View::make(
            'veracore::create_order',
            array_merge(
                $payload,
                $this->getAuthenticationPayload()
            )
        )->render();

        if (! $this->validateXmlPayload($payload)) {
            throw new RuntimeException(
                'The XML payload for the createOrder API call is invalid.'
            );
        }

        $url = $this->makeUrl(self::BASE_ENDPOINT);

        $response = $this->post(
            url: $url,
            body: $payload,
            bodyContentType: 'text/xml',
        );

        if ($response->successful()) {
            // Extract returned ID and return it.
            $desiredXPaths = [
                '//*[local-name()="OrderID"]',
            ];
            $parsedXml = $this->parseResponseXPaths(
                $response->body(),
                $desiredXPaths,
            );

            return (int) $parsedXml[$desiredXPaths[0]];
        } else {

            $errorPaths = '//*/faultstring';
            $extractedText = $response->reason();

            $parsedXml = $this->parseResponseXPaths($response->body(), [$errorPaths]);
            if (! empty($parsedXml[$errorPaths])) {

                $error = $parsedXml[$errorPaths];
                $exceptionPosition = strpos($error, 'System.Exception');

                if ($exceptionPosition !== false) {
                    // Extract the text after "System.Exception"
                    $afterException = substr($error, $exceptionPosition + strlen('System.Exception'));

                    // Find the position of "at ProMail..." in the extracted text
                    $proMailPosition = strpos($afterException, 'at ProMail.');

                    if ($proMailPosition !== false) {
                        // Get the text before "at ProMail..."
                        $extractedText = trim(ltrim(substr($afterException, 0, $proMailPosition), ':')).'.';
                    }
                }

            }

            // Error handling
            throw new RuntimeException(
                'Error submitting order to Veracore: '.$extractedText
            );
        }

    }

    /**
     * @throws Throwable
     * @throws VeracoreCannotFetchOrderException
     */
    private function findExistingOrderInVeracore(int $id, string $reference): ?array
    {
        try {
            $order = $this->getOrder($id);
        } catch (IntegrationTimeoutException $e) {
            // If the request times out, we throw a custom exception so that it can be handled later.
            throw new VeracoreCannotFetchOrderException($e->getMessage());
        } catch (Exception) {
            // For all other exceptions, we take it as the order doesn't exist.
            return null;
        }

        return ! empty(@$order['OrdHead']['ReferenceNo']) &&
                $order['OrdHead']['ReferenceNo'] == $reference ? $order['OrdHead'] : null;
    }

    /**
     * @throws Throwable
     */
    private function getAccessToken(): string
    {
        try {
            return $this->credential->getAccessToken();
        } catch (RuntimeException|Throwable) {
            // Request for access token and return it.
            return $this->requestFreshToken();
        }
    }

    /**
     * @throws Throwable
     */
    private function requestFreshToken(): string
    {

        $url = $this->makeUrl(self::ACCESS_TOKEN_ENDPOINT);
        $headers = ['Content-Type' => 'application/json', 'Accept' => 'application/json'];

        $payload = array_merge($this->getAuthenticationPayload(), ['systemId' => $this->credential->getSystemId()]);

        $response = $this->post(
            url: $url,
            body: $payload,
            headers: $headers,
            tries: 2
        );

        if ($response->successful()) {
            return $this->credential->updateAccessToken($response->json());
        } else {
            throw new RuntimeException(
                'Error getting access token from Veracore: '.$response->reason()
            );
        }
    }

    private function getAuthenticationPayload(): array
    {
        return [
            'username' => $this->credential->getUsername(),
            'password' => $this->credential->getPassword(),
        ];
    }

    /**
     * @throws Throwable
     */
    public function cancelOrder(string|int $veracoreOrderId): void
    {
        $payload = View::make(
            'veracore::cancel_order',
            array_merge(
                $this->getAuthenticationPayload(),
                ['id' => $veracoreOrderId]
            )
        )->render();

        if (! $this->validateXmlPayload($payload)) {
            throw new RuntimeException(
                'The XML payload for the cancelOrder API call is invalid.'
            );
        }

        $url = $this->makeUrl(self::ORDER_ADJUSTMENTS_ENDPOINT);

        $response = $this->post(
            url: $url,
            body: $payload,
            bodyContentType: 'text/xml',
            headers: ['SOAPAction' => 'http://omscom/CancelOrder'],
            tries: 3
        );

        // Indicate in api logs and throw an error if order cancellation fails
        // Note that Veracore returns a 200 OK even if the request fails.
        // We need to check the response body to see if there were any errors.
        $errorPath = '//*[local-name()="ExceptionMessage"]';
        $parsedXml = $this->parseResponseXPaths($response->body(), [$errorPath]);
        if (! empty($parsedXml[$errorPath])) {
            $error = $parsedXml[$errorPath];

            dispatch(new UpdateApiLogJob(
                uuid: $this->currentLogUuid,
                statusCode: 500,
                responseHeaders: $response->headers(),
                responseBody: $response->body()
            ));

            throw new RuntimeException(
                'Error cancelling order to Veracore: '.$error
            );
        }
    }

    /**
     * @throws Throwable
     */
    public function addLinesToFulfillmentOrder(int $veracoreId, array $fulfillmentLines): void
    {
        $payload = View::make(
            'veracore::add_lines_to_order',
            array_merge(
                $this->getAuthenticationPayload(),
                ['id' => $veracoreId],
                ['fulfillmentLines' => $fulfillmentLines]
            )
        )->render();

        if (! $this->validateXmlPayload($payload)) {
            throw new RuntimeException(
                'The XML payload for the Add Offer API call is invalid.'
            );
        }

        $url = $this->makeUrl(self::ORDER_ADJUSTMENTS_ENDPOINT);

        $response = $this->post(
            url: $url,
            body: $payload,
            bodyContentType: 'text/xml',
            headers: ['SOAPAction' => 'http://omscom/AddOffer'],
            tries: 3
        );

        if (! $response->successful()) {
            throw new RuntimeException(
                'Error adding fulfillment lines to Veracore: '.$response->reason()
            );
        }
    }

    /**
     * @throws Throwable
     */
    public function updateFulfillmentLineQuantities(int $veracoreId, array $changingFulfillmentLines): void
    {
        $increasingLines = collect($changingFulfillmentLines)->where('quantity_difference', '>', 0)->toArray();
        if (! empty($increasingLines)) {
            $this->updateVeracoreLineQuantities($veracoreId, $increasingLines, 'veracore::increase_line_quantity');
        }

        $decreasingLines = collect($changingFulfillmentLines)->where('quantity_difference', '<', 0)->toArray();
        if (! empty($decreasingLines)) {
            $this->updateVeracoreLineQuantities($veracoreId, $decreasingLines, 'veracore::decrease_line_quantity');
        }

    }

    /**
     * @throws Throwable
     */
    private function updateVeracoreLineQuantities(int $veracoreId, array $changingLines, string $template): void
    {
        $payload = View::make(
            $template,
            array_merge(
                $this->getAuthenticationPayload(),
                ['id' => $veracoreId],
                ['changingLines' => $changingLines]
            )
        )->render();

        if (! $this->validateXmlPayload($payload)) {
            throw new RuntimeException(
                'The XML payload for the Add Offer API call is invalid.'
            );
        }

        $url = $this->makeUrl(self::ORDER_ADJUSTMENTS_ENDPOINT);

        $response = $this->post(
            url: $url,
            body: $payload,
            bodyContentType: 'text/xml',
            headers: ['SOAPAction' => 'http://omscom/UpdateOfferQuantity'],
            tries: 3
        );

        // Veracore returns a 200 OK even if the request fails.
        // We need to check the response body to see if there were any errors.
        $errorPaths = '//*[local-name()="ExceptionMessage"]';

        $parsedXml = $this->parseResponseXPaths($response->body(), [$errorPaths]);
        if (! empty($parsedXml[$errorPaths])) {

            $extractedText = $parsedXml[$errorPaths];
            throw new RuntimeException(
                'Error updating fulfillment line quantities on Veracore: '.$extractedText
            );
        }

    }

    private function buildQuery($params): string
    {
        return urldecode(http_build_query($params));
    }
}
