<?php

namespace Modules\Veracore\Repositories;

use App\Models\SalesOrderFulfillment;
use App\Models\ShippingMethod;
use App\Services\ShippingProvider\ShippingProviderOrder;
use App\Services\ShippingProvider\ShippingProviderRepository;
use Illuminate\Database\Eloquent\Model;
use Modules\Veracore\Entities\VeracoreOrder;


class VeracoreOrderRepository implements ShippingProviderRepository
{

    /**
     * @param  int|string  $response
     * @param  SalesOrderFulfillment  $fulfillment
     * @return VeracoreOrder|Model
     */
    public function updateOrCreateByResponse(int|string $response, SalesOrderFulfillment $fulfillment): VeracoreOrder|Model
    {
        return VeracoreOrder::query()
            ->firstOrCreate(['veracore_id' => $response], [
                'json_data' => [
                    'Order' => [
                        'OrderID' => $response,
                        'ReferenceNumber' => $fulfillment->fulfillment_number,
                    ]
                ],
                'sku_fulfillment_id' => $fulfillment->id,
            ]);
    }

    /**
     * @param  array  $orders
     * @return array
     */
    public function getFulfillmentsForOrders(array $orders): array
    {
        return SalesOrderFulfillment::query()
            ->select('sales_order_fulfillments.*')
            ->addSelect('veracore_id as shipping_provider_id')
            ->join('veracore_orders', 'veracore_orders.sku_fulfillment_id', 'sales_order_fulfillments.id')
            ->whereIn('veracore_orders.veracore_id', collect($orders)->pluck('veracore_id')->toArray())
            ->get()->all();
    }

    /**
     * @param  array  $ids
     * @return array
     */
    public function findByIds(array $ids): array
    {
        return VeracoreOrder::query()
            ->whereIn('veracore_id', $ids)
            ->get()->all();
    }


    /**
     * @param array $orders
     * @return array
     */
    public function updateAll(array $orders): array
    {
        array_walk_recursive($orders, function(&$value){
            if (is_string($value)) {
                $value = str_replace(["\r\n", "\n"], "\\n", $value);
            }
        });

        // Only saving orders that originated from SKU
        $orderIds = collect($orders)->pluck('Order.OrderID')->toArray();
        batch()->update(
            new VeracoreOrder,
            array_map(function($order){
                return ['json_data' => json_encode($order), 'veracore_id' => data_get($order, 'Order.OrderID')];
            }, $orders),
            'veracore_id'
        );

        return $this->findByIds($orderIds);
    }

    /**
     * @return array
     */
    public function getOrderIdsNeedingTrackingInfo(): array
    {
        return VeracoreOrder::query()
            ->whereNull('json_data')
            ->orWhereRaw("JSON_EXTRACT(json_data, '$.Shipping.TrackingNumber') IS NULL")
            ->pluck('veracore_id')
            ->all();
    }

    /**
     * @param  array  $shipments
     * @return array
     */
    public function getShippingMethodsIn(array $shipments): array
    {
        $shippingMethodNames = array_unique(
            array_map(
                fn($shipment) => "{$shipment['FreightCarrier']} {$shipment['FreightCodeDescription']}",
                $shipments
            )
        );

        return ShippingMethod::query()
            ->whereIn('full_name', $shippingMethodNames)->get()->toArray();
    }

    /**
     * @param  array  $fulfillmentData
     * @return int
     */
    public function updateFulfillmentsWithTrackingInfo(array $fulfillmentData): int
    {
        return batch()->update(new SalesOrderFulfillment, $fulfillmentData, 'id');
    }

    /**
     * @param int $fulfillmentId
     * @return VeracoreOrder|Model|null
     */
    public function getOrderByFulfillmentId(int $fulfillmentId): VeracoreOrder|Model|null
    {
        return VeracoreOrder::query()
            ->where('sku_fulfillment_id', $fulfillmentId)
            ->first();
    }

    /**
     * @param  VeracoreOrder|ShippingProviderOrder  $order
     * @param  string  $reference
     * @return void
     */
    public function setReferenceNumber(VeracoreOrder|ShippingProviderOrder $order, string $reference): void
    {
        $data = $order->json_data;
        $data['Order']['ReferenceNumber'] = $reference;
        $order->json_data = $data;
        $order->save();
    }
}