<?php

namespace Modules\Veracore\Entities;

use App\Data\ShippingProviderTrackingInfoData;
use App\Models\SalesOrderFulfillment;
use App\Services\ShippingProvider\ShippingProviderOrder;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Modules\Veracore\Database\factories\VeracoreOrderFactory;

/**
 * @property int $id
 * @property int $veracore_id
 * @property array $json_data
 * @property int $sku_fulfillment_id
 * @property bool $notified_about_cancellation
 * @property SalesOrderFulfillment $salesOrderFulfillment
 * @property Carbon $created_at
 * @property Carbon $updated_at
 */
class VeracoreOrder extends ShippingProviderOrder
{

    use HasFactory;

    /**
     * @var string[]
     */
    protected $fillable = [
        'json_data',
        'sku_fulfillment_id',
        'notified_about_cancellation'
    ];

    /**
     * @var string[]
     */
    protected $casts = [
        'json_data' => 'array',
        'notified_about_cancellation' => 'bool'
    ];


    /**
     * @return BelongsTo
     */
    public function salesOrderFulfillment(): BelongsTo{
        return $this->belongsTo(SalesOrderFulfillment::class, 'sku_fulfillment_id');
    }


    /**
     * @return VeracoreOrderFactory
     */
    protected static function newFactory()
    {
        return VeracoreOrderFactory::new();
    }

    /**
     * @return string|int
     */
    public function getOrderId(): string|int
    {
        return $this->veracore_id;
    }

    /**
     * @return ShippingProviderTrackingInfoData
     */
    public function getTrackingInfo(): ShippingProviderTrackingInfoData
    {
        $carrier = data_get($this->json_data, 'Shipping.FreightCarrier');
        $description = data_get($this->json_data, 'Shipping.FreightCodeDescription');

        $carrierDescription = $carrier ? $carrier . ' ' . $description : $description;

        return new ShippingProviderTrackingInfoData(
            $this->json_data['Shipping']['TrackingNumber'],
            $carrierDescription,
            (float) $this->json_data['Shipping']['Freight']['MarkedUpFreight'] ?? 0,
            Carbon::parse(data_get($this->json_data, 'Shipping.UTCShippedDateTime'), 'utc'),
            null
        );
    }


}