<?php

namespace Modules\ShipMyOrders\Tests\Feature;

use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\ApiLog;
use App\Models\Customer;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\Product;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\User;
use App\Models\Warehouse;
use App\Repositories\SalesOrder\SalesOrderRepository;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Http;
use Modules\ShipMyOrders\Entities\ShipMyOrdersOrder;
use Modules\ShipMyOrders\Managers\ShipMyOrdersDispatchManager;
use Modules\ShipMyOrders\Managers\ShipMyOrdersOrderManager;
use Modules\ShipMyOrders\Services\ShipMyOrdersClient;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class ShipMyOrdersOrderTest extends TestCase
{

    use FastRefreshDatabase;
    use WithFaker;

    private IntegrationInstance $instance;
    private string $baseUrl;
    private string $apiPath;

    public function setUp(): void
    {
        parent::setUp();

        $integration = Integration::query()->where('name',Integration::NAME_SHIPMYORDERS)->first();
        $warehouse = Warehouse::factory()->create(['type' => Warehouse::TYPE_3PL]);

        $this->instance = IntegrationInstance::factory()->create([
            'name' => Integration::NAME_SHIPMYORDERS,
            'connection_settings' => [
                'username' => $this->faker->md5(),
                'password' => $this->faker->md5(),
                'clientId' => $this->faker->word()
            ],
            'integration_id' => $integration ? $integration->id : Integration::factory()->create([
                'name' => Integration::NAME_SHIPMYORDERS,
                'integration_type' => Integration::TYPE_SHIPPING_PROVIDER
            ]),
            'integration_settings' => [
                'linked_warehouse_id' => $warehouse->id,
            ]
        ]);

        $this->actingAs(User::factory()->create());

        $this->baseUrl = config('shipmyorders.api_url');
        $this->apiPath = config('shipmyorders.api_path');

        Http::preventStrayRequests();
    }

    public function test_it_records_api_log_when_smo_fulfillment_fails(){

        $fulfillment = SalesOrderFulfillment::factory()->create([
            'fulfillment_type' => SalesOrderFulfillment::TYPE_SHIPMYORDERS,
            'status' => SalesOrderFulfillment::STATUS_SUBMITTED,
            'sales_order_id' => SalesOrder::factory()->create()->id
        ]);

        ShipMyOrdersOrder::factory()->create(
            [
                'json_data' => [
                    'Order' => [
                        'OrderId' => 123
                    ]
                ],
                'sku_fulfillment_id' => $fulfillment->id,
                'reference_number' => $fulfillment->fulfillment_number
            ]
        );

        Http::fake([
            ShipMyOrdersClient::makeUrl(ShipMyOrdersClient::GETTRACKING) => Http::response(
                "Internal Server Error", 500
            )
        ]);

        /** @var ShipMyOrdersOrderManager $manager */
        $manager = app(ShipMyOrdersOrderManager::class);
        $manager->updateTrackingInfo();

        // Api log should be created
        $this->assertDatabaseCount((new ApiLog())->getTable(), 1);
    }

    public function test_it_can_get_smo_tracking_info(): void{

        // Prepare orders
        $id = '100';

        $integrationInstance = IntegrationInstance::factory()->create([
            'name' => Integration::NAME_SHOPIFY,
            'integration_id' => Integration::query()->firstOrCreate(
                ['name' => Integration::NAME_SHOPIFY],
                ['integration_type' => Integration::TYPE_SALES_CHANNEL]
            )->id,
            'is_automatic_sync_enabled' => true,
        ]);
        /** @var SalesChannel $salesChannel */
        $salesChannel = SalesChannel::factory()->create([
            'integration_instance_id' => $integrationInstance->id
        ]);

        $salesOrder = SalesOrder::factory()->create([
            'sales_channel_id' => $salesChannel->id,
        ]);

        $fulfillment = SalesOrderFulfillment::factory()->create([
            'fulfillment_type' => SalesOrderFulfillment::TYPE_SHIPMYORDERS,
            'status' => SalesOrderFulfillment::STATUS_SUBMITTED,
            'sales_order_id' => $salesOrder->id
        ]);

        ShipMyOrdersOrder::factory()->create(
            [
                'json_data' => [
                    'ID' => $id
                ],
                'sku_fulfillment_id' => $fulfillment->id,
                'reference_number' => $fulfillment->fulfillment_number
            ]
        );


        $tracking = $this->faker->unique()->md5();

        // Mock endpoints
        $now = now()->format('m/d/Y');
        Http::fake([
            ShipMyOrdersClient::makeUrl(ShipMyOrdersClient::GETTRACKING) => Http::response(
                "
                    <GetTrackingResponse>
                        <Tracking>$tracking</Tracking>
                        <Carrier>USPS Priority Mail</Carrier>
                        <ShipDate>$now</ShipDate>
                    </GetTrackingResponse>
                ", 201
            )
        ]);


        // Download orders
        /** @var ShipMyOrdersOrderManager $manager */
        $manager = app(ShipMyOrdersOrderManager::class);
        $manager->updateTrackingInfo();

        // Assert
        $this->assertDatabaseCount((new ShipMyOrdersOrder())->getTable(), 1);
        $this->assertDatabaseHas((new ShipMyOrdersOrder())->getTable(), [
            'smo_id' => $id
        ]);

        // Tracking should be updated
        $this->assertDatabaseHas((new SalesOrderFulfillment())->getTable(), [
            'id' => $fulfillment->id,
            'tracking_number' => $tracking,
            'status' => SalesOrderFulfillment::STATUS_FULFILLED
        ]);

        // Sales order should be fulfilled and closed.
        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'id' => $salesOrder->id,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
            'order_status' => SalesOrder::STATUS_CLOSED
        ]);

        $this->assertDatabaseCount((new ApiLog())->getTable(), 1);

    }

    public function test_it_dispatches_orders_to_smo(): void{

        // Prepare
        $product = Product::factory()->create();
        $warehouse = Warehouse::query()
            ->firstOrCreate(['type' => Warehouse::TYPE_3PL])
            ->withDefaultLocation();
        $this->postJson(route('inventory-adjustments.store'), [
            'product_id' => $product->id,
            'adjustment_date' => now(),
            'quantity' => 3,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 10
        ])->assertSuccessful();
        $response = $this->postJson(route('sales-orders.store'), [
            'currency_code' => 'USD',
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'deliver_by_date' => now()->addDays(2),
            'customer_id' => Customer::factory()->create()->id,
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 3,
                    'description' => $this->faker->sentence(),
                    'amount' => 5,
                    'warehouse_id' => $warehouse->id,
                ]
            ]
        ])->assertSuccessful();


        Http::fake([
            $this->baseUrl . $this->apiPath . 'createorder' => Http::response(
                '
                <CreateOrder>
                    <Result>success</Result>
                    <ID>2705375</ID>
                </CreateOrder>
                ', 201)
        ]);

        $orders = new SalesOrderRepository;
        $fulfillment = $orders->createFulfillment(
            order: $orders->findById($response->json('data.id')),
            payload: [
                'fulfillment_type' => SalesOrderFulfillment::TYPE_SHIPMYORDERS,
                'fulfilled_at' => now(),
                'warehouse_id' => $warehouse->id,
                'fulfillment_lines' => [
                    [
                        'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
                        'quantity' => 3
                    ]
                ]
            ]
        );

        // Action
        $dispatcher = app(ShipMyOrdersDispatchManager::class);
        $dispatcher->dispatchFulfillmentToProvider($fulfillment);

        // Assertions
        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'id' => $response->json('data.id'),
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_AWAITING_TRACKING,
        ]);

        $this->assertDatabaseHas((new SalesOrderFulfillment())->getTable(), [
            'id' => $fulfillment->id,
            'status' => SalesOrderFulfillment::STATUS_SUBMITTED
        ]);

        $this->assertDatabaseHas((new ShipMyOrdersOrder())->getTable(), [
            'sku_fulfillment_id' => $fulfillment->id,
            'reference_number' => $fulfillment->fulfillment_number
        ]);

        $this->assertDatabaseCount((new ApiLog())->getTable(), 1);
    }

}
