<?php

namespace Modules\ShipMyOrders\Tests\Feature;

use App\Models\NominalCode;
use Illuminate\Foundation\Testing\WithFaker;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInvoice;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInvoiceLine;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInvoiceNominalCodeMappingRule;
use Modules\ShipMyOrders\Managers\ShipMyOrdersInvoiceNominalCodeMappingRuleManager;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class ShipMyOrdersInvoiceNominalCodeMappingTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    private ShipMyOrdersInvoiceNominalCodeMappingRuleManager $manager;

    protected function setUp(): void
    {
        parent::setUp();
        $this->manager = $this->app->make(ShipMyOrdersInvoiceNominalCodeMappingRuleManager::class);
    }

    public function test_can_get_nominal_code_by_condition_quantity(): void
    {
        ShipMyOrdersInvoiceNominalCodeMappingRule::factory()->create([
            'nominal_code_id' => NominalCode::factory()->create(['code' => 1000])->id,
            'conditions' => [
                [
                    'column' => 'quantity',
                    'operator' => '>',
                    'value' => 5,
                ]
            ],
        ]);

        $invoice = ShipMyOrdersInvoice::factory()->create();

        $line1 = ShipMyOrdersInvoiceLine::factory()->create(['quantity' => 10, 'ship_my_orders_invoice_id' => $invoice->id]);
        $line2 = ShipMyOrdersInvoiceLine::factory()->create(['quantity' => 4, 'ship_my_orders_invoice_id' => $invoice->id]);

        // test for line 1
        $nominalCode = $this->manager->getNominalCodeForInvoiceLine($line1);
        $this->assertNotNull($nominalCode);
        $this->assertEquals('1000', $nominalCode->code);

        // test for line 2
        $nominalCode = $this->manager->getNominalCodeForInvoiceLine($line2);
        $this->assertNull($nominalCode);
    }

    public function test_can_get_nominal_code_by_condition_description(): void
    {
        // test using description contains
        $rule = ShipMyOrdersInvoiceNominalCodeMappingRule::factory()->create([
            'nominal_code_id' => NominalCode::factory()->create(['code' => 1000])->id,
            'conditions' => [
                [
                    'column' => 'description',
                    'operator' => 'contains',
                    'value' => 'USPS',
                ]
            ],
        ]);

        $invoice = ShipMyOrdersInvoice::factory()->create();

        $line1 = ShipMyOrdersInvoiceLine::factory()->create(['description' => 'USPS Priority Mail', 'ship_my_orders_invoice_id' => $invoice->id]);
        $line2 = ShipMyOrdersInvoiceLine::factory()->create(['description' => 'Fulfillment Fee 1 item', 'ship_my_orders_invoice_id' => $invoice->id]);

        // test for line 1
        $nominalCode = $this->manager->getNominalCodeForInvoiceLine($line1);
        $this->assertNotNull($nominalCode);
        $this->assertEquals('1000', $nominalCode->code);

        // test for line 2
        $nominalCode = $this->manager->getNominalCodeForInvoiceLine($line2);
        $this->assertNull($nominalCode);

        // test using description not contains
        $rule->update([
            'conditions' => [
                [
                    'column' => 'description',
                    'operator' => 'not contains',
                    'value' => 'USPS',
                ]
            ],
        ]);

        // test for line 1
        $nominalCode = $this->manager->getNominalCodeForInvoiceLine($line1);
        $this->assertNull($nominalCode);

        // test for line 2
        $nominalCode = $this->manager->getNominalCodeForInvoiceLine($line2);
        $this->assertNotNull($nominalCode);
        $this->assertEquals('1000', $nominalCode->code);

    }
}