<?php

namespace Modules\ShipMyOrders\Tests\Feature;

use App\Models\Integration;
use App\Models\User;
use App\Models\Warehouse;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Testing\WithFaker;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class ShipMyOrdersIntegrationInstanceControllerTest extends TestCase
{

    use FastRefreshDatabase;
    use WithFaker;

    protected Warehouse $warehouse;
    protected Integration|Model $integration;
    
    protected function setUp(): void
    {
        parent::setUp();
        $this->actingAs(User::factory()->create());
        $this->warehouse = Warehouse::factory()->create();
        $this->integration = Integration::query()->firstOrCreate([
            'name' => Integration::NAME_SHIPMYORDERS
        ]);
    }

    public function test_it_can_create_smo_integration_instance(): void{

        $this->postJson(route('shipmyorders.integration-instances.store'), [
            'integration_id' => Integration::factory()->create()->id,
            'name' => 'SMO',
            'integration_settings' => [
                'connection_name' => 'SMO',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'client_id' => $this->faker->uuid(),
            ]
        ])->assertSuccessful()->assertJsonStructure([
            'data' => [
                'id',
                'integration_id',
                'name',
                'integration_settings' => [
                    'connection_name',
                    'linked_warehouse',
                    'automate_fulfillment'
                ],
                'connection_settings',
                'created_at',
                'updated_at'
            ]
        ]);

        $this->assertDatabaseHas('integration_instances', [
            'name' => 'SMO',
        ]);

    }

    public function test_it_can_get_smo_integration_instance(): void{

            $response = $this->postJson(route('shipmyorders.integration-instances.store'), [
                'integration_id' => Integration::factory()->create()->id,
                'name' => 'SMO',
                'integration_settings' => [
                    'connection_name' => 'SMO',
                    'linked_warehouse_id' => $this->warehouse->id,
                    'automate_fulfillment' => true
                ],
                'connection_settings' => [
                    'username' => $this->faker->userName(),
                    'password' => $this->faker->password(),
                    'client_id' => $this->faker->uuid(),
                ]
            ])->assertSuccessful();

            $this->getJson(route('shipmyorders.integration-instances.show', $response->json('data.id')))
                ->assertSuccessful()->assertJsonStructure([
                    'data' => [
                        'id',
                        'integration_id',
                        'name',
                        'integration_settings' => [
                            'connection_name',
                            'linked_warehouse',
                            'automate_fulfillment'
                        ],
                        'connection_settings',
                        'created_at',
                        'updated_at'
                    ]
                ]);
    }

    public function test_it_cannot_create_smo_instance_without_linked_warehouse_id(): void{

        $this->postJson(route('shipmyorders.integration-instances.store'), [
            'integration_id' => Integration::factory()->create()->id,
            'name' => 'SMO',
            'integration_settings' => [
                'connection_name' => 'SMO',
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'client_id' => $this->faker->uuid(),
            ]
        ])->assertStatus(422);
    }

    public function test_it_cannot_create_multiple_smo_instances(): void{

        $this->postJson(route('shipmyorders.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'SMO',
            'integration_settings' => [
                'connection_name' => 'SMO',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'client_id' => $this->faker->uuid(),
            ]
        ])->assertSuccessful();

        $this->postJson(route('shipmyorders.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'SMO',
            'integration_settings' => [
                'connection_name' => 'SMO',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'client_id' => $this->faker->uuid(),
            ]
        ])->assertStatus(422);
    }


    public function test_it_can_update_smo_integration_instance(): void{

        $response = $this->postJson(route('shipmyorders.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'SMO',
            'integration_settings' => [
                'connection_name' => 'SMO',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'client_id' => $this->faker->uuid(),
            ]
        ])->assertSuccessful();

        $this->putJson(route('shipmyorders.integration-instances.update', $response->json('data.id')), [
            'integration_id' => $this->integration->id,
            'name' => 'SMO',
            'integration_settings' => [
                'connection_name' => 'SMO',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'client_id' => $this->faker->uuid(),
            ]
        ])->assertSuccessful()->assertJsonStructure([
            'data' => [
                'id',
                'integration_id',
                'name',
                'integration_settings' => [
                    'connection_name',
                    'linked_warehouse',
                    'automate_fulfillment'
                ],
                'connection_settings',
                'created_at',
                'updated_at'
            ]
        ]);

        $this->assertDatabaseHas('integration_instances', [
            'name' => 'SMO',
        ]);

    }

    public function test_it_can_change_name_for_same_smo_instance(): void{

        $response = $this->postJson(route('shipmyorders.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'SMO',
            'integration_settings' => [
                'connection_name' => 'SMO',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'client_id' => $this->faker->uuid(),
            ]
        ])->assertSuccessful();

        $this->putJson(route('shipmyorders.integration-instances.update', $response->json('data.id')), [
            'integration_id' => $this->integration->id,
            'name' => 'SMO2',
            'integration_settings' => [
                'connection_name' => 'SMO',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'client_id' => $this->faker->uuid(),
            ]
        ])->assertSuccessful()->assertJsonStructure([
            'data' => [
                'id',
                'integration_id',
                'name',
                'integration_settings',
                'connection_settings',
                'created_at',
                'updated_at'
            ]
        ]);

        $this->assertDatabaseHas('integration_instances', [
            'name' => 'SMO2',
        ]);
    }

    public function test_it_can_delete_smo_instance(): void{

        $response = $this->postJson(route('shipmyorders.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'SMO',
            'integration_settings' => [
                'connection_name' => 'SMO',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'client_id' => $this->faker->uuid(),
            ]
        ])->assertSuccessful();

        $this->deleteJson(route('shipmyorders.integration-instances.destroy', $response->json('data.id')))
            ->assertSuccessful();

        $this->assertDatabaseMissing('integration_instances', [
            'name' => 'SMO',
        ]);
    }
    
    public function test_it_can_change_linked_direct_warehouse_to_3pl_warehouse(): void{

        $this->postJson(route('shipmyorders.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'SMO',
            'integration_settings' => [
                'connection_name' => 'SMO',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'client_id' => $this->faker->uuid(),
            ]
        ])->assertSuccessful();


        $this->assertDatabaseHas('warehouses', [
            'id' => $this->warehouse->id,
            'type' => Warehouse::TYPE_3PL
        ]);
    }

    public function test_only_direct_or_3pl_warehouse_can_be_linked(): void{

        $this->postJson(route('shipmyorders.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'SMO',
            'integration_settings' => [
                'connection_name' => 'SMO',
                'linked_warehouse_id' => Warehouse::factory()->create([
                    'type' => Warehouse::TYPE_SUPPLIER
                ])->id,
                'automate_fulfillment' => true
            ],
            'connection_settings' => [
                'username' => $this->faker->userName(),
                'password' => $this->faker->password(),
                'client_id' => $this->faker->uuid(),
            ]
        ])->assertStatus(422);
    }

    public function test_it_does_not_change_is_automatic_sync_enabled_setting_when_connection_settings_change(): void{

        $response = $this->postJson(route('shipmyorders.integration-instances.store'), [
            'integration_id' => $this->integration->id,
            'name' => 'SMO',
            'integration_settings' => [
                'connection_name' => 'SMO',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => false
            ],
            'connection_settings' => [
                'username' => 'test',
                'password' => 'test',
                'systemId' => 'test',
            ],
            'is_automatic_sync_enabled' => true
        ])->assertSuccessful();

        $this->putJson(route('shipmyorders.integration-instances.update', $response->json('data.id')), [
            'integration_id' => $this->integration->id,
            'name' => 'SMO',
            'integration_settings' => [
                'connection_name' => 'SMO',
                'linked_warehouse_id' => $this->warehouse->id,
                'automate_fulfillment' => false
            ],
            'connection_settings' => [
                'username' => 'test2',
                'password' => 'test2',
                'systemId' => 'test2',
            ]
        ])->assertSuccessful()->assertJsonStructure([
            'data' => [
                'id',
                'integration_id',
                'name',
                'integration_settings',
                'connection_settings',
                'created_at',
                'updated_at'
            ]
        ]);

        $this->assertDatabaseHas('integration_instances', [
            'name' => 'SMO',
            'is_automatic_sync_enabled' => true
        ]);

    }

}
